import * as API from '../../../api/sensors';

const START_TO_LOAD_LABELS = 'START_TO_LOAD_LABELS';
const FINISH_TO_LOAD_LABELS = 'FINISH_TO_LOAD_LABELS';
const FAILED_TO_LOAD_LABELS = 'FAILED_TO_LOAD_LABELS';
const CLEAR_LABELS = 'CLEAR_LABELS';

const DEFAULT_SENSOR_NAMES = { names: [], truncated: false };

export const fetchLabels = (projectId, params, useNewFormat) => (dispatch) => {
  dispatch({ type: START_TO_LOAD_LABELS });

  let promise;

  if (params.names === 'cluster,service') {
    promise = API.fetchClusterServices(projectId, params).then((resp) => ({
      ...resp,
      sensorNames: DEFAULT_SENSOR_NAMES,
    }));
  } else if (useNewFormat) {
    const sensorNamesPromise = API.fetchSensorNamesInNewFormat(projectId, params);
    const labelValuesPromise = API.fetchLabelValuesInNewFormat(projectId, params);

    promise = Promise.all([sensorNamesPromise, labelValuesPromise]).then((resp) => {
      if (resp && resp.length && resp.length === 2) {
        const sensorNamesResp = resp[0];
        const labelValuesResp = resp[1];

        return {
          ...labelValuesResp,
          sensorNames: sensorNamesResp,
        };
      }

      return {
        sensorNames: DEFAULT_SENSOR_NAMES,
        maxCount: 0,
        sensorsCountByCluster: {},
        labels: {},
      };
    });
  } else {
    promise = API.fetchLabels(projectId, params).then((resp) => ({
      ...resp,
      sensorNames: DEFAULT_SENSOR_NAMES,
    }));
  }

  return promise
    .then(
      (resp) => dispatch({ type: FINISH_TO_LOAD_LABELS, payload: resp }),
      (resp) => dispatch({ type: FAILED_TO_LOAD_LABELS, payload: resp.message }),
    );
};

export const clearLabels = () => (dispatch) => dispatch({ type: CLEAR_LABELS });

const DEFAULT_SENSOR_LABELS_STATE = { loading: false, data: null, error: null };

export const sensorLabelsReducer = (state = DEFAULT_SENSOR_LABELS_STATE, action) => {
  switch (action.type) {
    case START_TO_LOAD_LABELS:
      return { loading: true, data: state.data, error: null };
    case FINISH_TO_LOAD_LABELS:
      return { loading: false, data: action.payload, error: null };
    case FAILED_TO_LOAD_LABELS:
      return { loading: false, data: null, error: action.payload };
    case CLEAR_LABELS:
      return DEFAULT_SENSOR_LABELS_STATE;
    default:
      return state;
  }
};
