import * as API from '../../../api/shards';
import { addDangerAlert, addInfoAlert } from '../toaster';
import { findCluster } from '../clusters/cluster';
import { findService } from '../services/service';
import * as INTERNAL_API from '../../../api/internal';
import { shortHostname } from '../../../utils/hosts';

const SHARD_FOUND = 'SHARD_FOUND';
const SHARD_SAVED = 'SHARD_SAVED';
const SHARD_ACTIVATED = 'SHARD_ACTIVATED';
const SHARD_DEACTIVATED = 'SHARD_DEACTIVATED';
const SHARD_STATE_UPDATED = 'SHARD_STATE_UPDATED';
const SHARD_DELETED = 'SHARD_DELETED';
const SHARD_CLEAR = 'SHARD_CLEAR';

export const findShard = (projectId, shardId) => (dispatch) =>
  API.fetchShard(projectId, shardId)
    .then(
      (resp) => dispatch({ type: SHARD_FOUND, payload: resp }),
      (resp) => dispatch(addDangerAlert(resp.message)),
    );

export const findShardWithRefs = (projectId, shardId) => (dispatch) =>
  dispatch(findShard(projectId, shardId))
    .then((action) => {
      if (action.type !== SHARD_FOUND) {
        return Promise.resolve();
      }
      const shard = action.payload;
      return Promise.all([
        dispatch(findCluster(projectId, shard.clusterId)),
        dispatch(findService(projectId, shard.serviceId)),
      ]);
    });

export const saveShard = (projectId, shard, isNew) => (dispatch) => {
  const promise = isNew
    ? API.createShard(projectId, shard)
    : API.updateShard(projectId, shard);
  promise.then(
    () => dispatch({ type: SHARD_SAVED }),
    (resp) => dispatch(addDangerAlert(resp.message)),
  );
  return promise;
};

export const activateShard = (shard) => (dispatch) => API.updateShard(shard.projectId, { ...shard, state: 'ACTIVE' })
  .then(
    (resp) => dispatch({ type: SHARD_ACTIVATED, payload: resp }),
    (resp) => dispatch(addDangerAlert(resp.message)),
  );

export const deactivateShard = (shard) => (dispatch) => API.updateShard(shard.projectId, { ...shard, state: 'INACTIVE' })
  .then(
    (resp) => dispatch({ type: SHARD_DEACTIVATED, payload: resp }),
    (resp) => dispatch(addDangerAlert(resp.message)),
  );

export const changeShardState = (shard, state) => (dispatch) =>
  API.updateShard(shard.projectId, { ...shard, state })
    .then(
      (resp) => dispatch({ type: SHARD_STATE_UPDATED, payload: resp }),
      (resp) => dispatch(addDangerAlert(resp.message)),
    );

export const deleteShard = (projectId, shardId) => (dispatch) =>
  API.deleteShard(projectId, shardId)
    .then(
      () => dispatch({ type: SHARD_DELETED }),
      (resp) => dispatch(addDangerAlert(resp.message)),
    );

export const reloadShard = (projectId, shardId, host, service) => (dispatch) =>
  INTERNAL_API.reloadShard(projectId, shardId, host, service)
    .then(
      () => dispatch(addInfoAlert(`${service} on ${shortHostname(host)} is reloading shard ${shardId}`)),
      (resp) => dispatch(addDangerAlert(resp.message)),
    );

export const clearShard = () => (dispatch) => dispatch({ type: SHARD_CLEAR });

export const shardReducer = (state = {}, action) => {
  switch (action.type) {
    case SHARD_FOUND:
    case SHARD_ACTIVATED:
    case SHARD_DEACTIVATED:
    case SHARD_STATE_UPDATED:
      return action.payload;
    case SHARD_SAVED:
    case SHARD_CLEAR:
    case SHARD_DELETED:
      return {};
    default:
      return state;
  }
};
