/* eslint-disable guard-for-in,no-restricted-syntax */
import Selectors from './Selectors';
import Selector from './Selector';
import UserLinksBasic from './UserLinksBasic';

/**
 * Create link to old graph from new selectors
 *
 * @param projectId
 * @param {String|Selectors} selectors
 * @param {String} forceCluster
 * @return {string} link to graph in old ui
 */
export function makeOldUiGraphUrl(projectId, selectors, forceCluster = '') {
  try {
    let parsedSelectors = selectors instanceof Selectors ? selectors : Selectors.parse(selectors);
    const params = new URLSearchParams();

    if (!parsedSelectors.hasKey('project')) {
      const newSelectorsList = [
        Selector.exact('project', projectId), ...parsedSelectors.getSelectors(),
      ];
      parsedSelectors = new Selectors(parsedSelectors.getNameSelector(), newSelectorsList);
    }

    const selectorsList = parsedSelectors.getSelectors();

    // Check that selector keys are unique
    const knownLabelKeys = {};
    for (let i = 0; i < selectorsList.length; ++i) {
      const key = selectorsList[i].getKey();
      if (key in knownLabelKeys) {
        return '';
      }
      knownLabelKeys[key] = true;
    }

    selectorsList.forEach((s) => {
      let key = s.getKey();

      if (key !== 'project' && key !== 'cluster' && key !== 'service') {
        key = `l.${key}`;
      }

      const op = s.getOp();
      let value = s.getValue();

      if (op === '=~' || op === '!~') {
        throw new Error('you cannot convert regex selectors to old legacy format');
      }

      if (op === '!=' || op === '!==') {
        value = `!${value}`;
      }

      params.append(key, value);
    });

    params.append('graph', 'auto');

    if (forceCluster) {
      params.append(UserLinksBasic.FORCE_CLUSTER_PARAM, forceCluster);
    }

    return `/?${params.toString()}`;
  } catch (e) {
    return '';
  }
}

/**
 * Add labels to selectors to support sub threshold-alert selectors
 *
 * @param {Selectors} selectors
 * @param {object} labels
 */
export function mergeSelectorsAndLabels(selectors, labels) {
  let result = selectors;

  for (const key in labels) {
    const value = labels[key] || '';
    result = result.addOverride(Selector.exact(key, value));
  }

  return result;
}

function exactOrAbsent(selector) {
  return selector.getOp() === '==' || (selector.getOp() === '=' && selector.getValue() === '-');
}

/**
 * @param {array} selectorsList list of <key, value> pairs in old format
 * @param {boolean} requiredBrackets true if brackets are required
 * @return {string} formatted selectors
 */
export function formatOldSelectorEntriesAsNewSelectors(selectorsList, requiredBrackets = false) {
  const parsedSelectorsList = selectorsList.map((s) => {
    const key = s[0];
    const value = s[1];

    if (value.startsWith('!')) {
      return Selector.notGlob(key, value.substring(1));
    }

    return Selector.glob(key, value);
  });

  return new Selectors('', parsedSelectorsList).format(requiredBrackets);
}

/**
 * Add labels to non-exact and not-absent selectors except to support sub expression-alert selectors
 *
 * @param {Selectors} selectors
 * @param {object} labels
 */
export function mergeProgramSelectorsAndLabels(selectors, labels) {
  const filteredLabels = { ...labels };

  const selectorsList = selectors.getSelectors();

  for (let i = 0; i < selectorsList.length; ++i) {
    const selector = selectorsList[i];
    const key = selector.getKey();
    if (labels[key]) {
      if (exactOrAbsent(selector)) {
        delete filteredLabels[key];
      }
    }
  }

  return mergeSelectorsAndLabels(selectors, filteredLabels);
}

/**
 * Parse selectors in old format and convert to new selectors
 *
 * @param {String} selectors
 * @return {Selectors} parsed selectors
 */
export function convertOldToNewSelectors(selectors) {
  const parsedSelectorsList = selectors.split('&')
    .filter(Boolean)
    .map((s) => {
      const eqIndex = s.indexOf('=');

      if (eqIndex > 0) {
        const key = s.substring(0, eqIndex);
        const value = s.substring(eqIndex + 1);
        if (value && value.startsWith('!')) {
          return Selector.notGlob(key, value.substr(1));
        }

        return Selector.glob(key, value);
      }

      return null;
    })
    .filter(Boolean);

  return Selectors.of('', ...parsedSelectorsList);
}

/**
 * Convert selectors to old format
 *
 * @param {Selectors} selectors
 * @return {String} selectors in old format
 */
export function convertSelectorsToOldFormat(selectors) {
  return selectors.getSelectors()
    .map((s) => {
      if (s.getOp() === '!=') {
        return `${s.getKey()}=!${s.getValue()}`;
      }
      return `${s.getKey()}=${s.getValue()}`;
    })
    .join('&');
}

export default {};
