import Selectors from './Selectors';
import Parser from './Parser';
import { isWhitespace } from './CharUtils';

export const SELECTORS_PART_TYPE = 'SELECTORS';
export const LOAD1_PART_TYPE = 'LOAD1';
export const LOAD_PART_TYPE = 'LOAD';
export const TEXT_PART_TYPE = 'TEXT';

const OLD_SELECTORS_REGEX = /(load|load1)\((["'])(.*?)(["'])\)/g;

function isSelectorNameChar(c) {
  return (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') || (c >= '0' && c <= '9') || c === '_' || c === '.';
}

function consumeBackQuotedName(code, pos, quote) {
  let curPos = pos - 1;
  while (curPos >= 0) {
    const c = code.charAt(curPos);
    if (c === quote) {
      if (curPos > 0 && code.charAt(curPos - 1) === '\\') {
        curPos -= 2;
      }
      break;
    }

    curPos -= 1;
  }

  return curPos + 1;
}

function consumeBackNameImpl(code, pos) {
  let curPos = pos;
  while (curPos >= 0) {
    const c = code.charAt(curPos);
    if (isSelectorNameChar(c)) {
      curPos -= 1;
    } else {
      break;
    }
  }

  if (curPos === pos) {
    return pos;
  }

  const name = code.substring(curPos + 1, pos + 1);

  if (name === 'return') {
    return pos;
  }

  return curPos + 1;
}

function consumeBackName(code, pos) {
  let curPos = pos - 1;

  while (curPos >= 0) {
    const c = code.charAt(curPos);
    if (isWhitespace(c)) {
      curPos -= 1;
    } else {
      break;
    }
  }

  if (curPos < 0) {
    return pos;
  }

  const c = code.charAt(curPos);
  if (c === '"' || c === '\'') {
    return consumeBackQuotedName(code, curPos);
  }

  const startPos = consumeBackNameImpl(code, curPos);
  if (startPos === curPos) {
    return pos;
  }

  return startPos;
}

function divideToNewSelectorsInCode(code) {
  const result = [];

  let lastIndex = 0;
  let lastSearchIndex = 0;

  for (;;) {
    const pos = code.indexOf('{', lastSearchIndex);

    if (pos >= 0) {
      try {
        const selectorsStartPos = consumeBackName(code, pos);
        const parser = new Parser(code, selectorsStartPos);
        Selectors.consumeSelectors(parser);
        const selectorsEndPos = parser._pos;
        if (selectorsEndPos === selectorsStartPos) {
          // Go to next '{'
          lastSearchIndex = pos + 1;
        } else {
          const textBefore = code.slice(lastIndex, selectorsStartPos);
          if (textBefore) {
            result.push({
              type: 'TEXT',
              text: textBefore,
            });
          }

          const selectorsText = code.slice(selectorsStartPos, selectorsEndPos);
          result.push({
            type: SELECTORS_PART_TYPE,
            text: selectorsText,
          });
          lastIndex = selectorsEndPos;
          lastSearchIndex = lastIndex;
        }
      } catch (e) {
        // Go to next "{"
        lastSearchIndex = pos + 1;
      }
    } else {
      const text = code.slice(lastIndex);

      if (text.length > 0) {
        result.push({
          text,
          type: 'TEXT',
        });
      }
      return result;
    }
  }
}

function divideToOldSelectorsInCode(code) {
  const result = [];

  let match;

  let lastIndex = 0;

  // eslint-disable-next-line no-cond-assign
  while ((match = OLD_SELECTORS_REGEX.exec(code))) {
    const matchedSubstr = match[0];
    const funcName = match[1];
    const selectors = match[3];
    const curIndex = match.index;

    const type = funcName === 'load' ? LOAD_PART_TYPE : LOAD1_PART_TYPE;

    const textBefore = code.substring(lastIndex, curIndex);
    if (textBefore) {
      result.push({
        type: TEXT_PART_TYPE,
        text: textBefore,
      });
    }

    lastIndex = curIndex + matchedSubstr.length;
    result.push({
      type,
      text: matchedSubstr,
      selectors,
    });
  }

  const lastTextPart = code.substring(lastIndex);
  if (lastTextPart) {
    result.push({
      type: TEXT_PART_TYPE,
      text: lastTextPart,
    });
  }

  return result;
}

/**
 * Divide code to parts with replaced selectors
 *
 * @param {String} code
 * @return {object[]} result of code division
 */
export function divideCodeBySelectors(code) {
  try {
    const result = [];

    const newSelectorsResult = divideToNewSelectorsInCode(code);

    for (let i = 0; i < newSelectorsResult.length; ++i) {
      const lexem = newSelectorsResult[i];
      if (lexem.type === TEXT_PART_TYPE) {
        const subResult = divideToOldSelectorsInCode(lexem.text);
        if (subResult.length > 0) {
          result.push(...subResult);
        }
      } else {
        result.push(lexem);
      }
    }

    return result;
  } catch (e) {
    console.error('failed to replace selectors in code', e);
    return {
      text: code,
      type: TEXT_PART_TYPE,
    };
  }
}

export function replaceSelectorsInCode(code, replacementFunc, params) {
  const parts = divideCodeBySelectors(code);

  const replacedParts = [];

  let allReplaced = true;

  for (let i = 0; i < parts.length; ++i) {
    const part = parts[i];
    if (part.type === TEXT_PART_TYPE) {
      replacedParts.push(part.text);
    } else {
      let replacementResult;
      if (part.type === SELECTORS_PART_TYPE) {
        replacementResult = replacementFunc(part.type, part.text, part.text, params);
      } else {
        replacementResult = replacementFunc(part.type, part.text, part.selectors, params);
      }
      allReplaced = allReplaced && replacementResult.isOk;
      replacedParts.push(replacementResult.part);
    }
  }

  return { result: replacedParts, allReplaced };
}

export default {};
