import Selectors from './Selectors';

const SHARD_KEYS = ['project', 'cluster', 'service'];

/**
 * Check that all shard selectors have exact values
 * Need to prevent pagination for cross-shard metabase requests
 *
 * @param {Selectors} selectors
 * @return true if all shard selectors have exact values,
 *         e.g.: project=junk, cluster=foo, service=bar
 */
export function isSingleShard(selectors) {
  const exactShardSelectors = selectors.getSelectors()
    .filter((s) => s.getKey() === 'project' || s.getKey() === 'cluster' || s.getKey() === 'service')
    .filter((s) => s.isExact());

  return !selectors.getNameSelector() && exactShardSelectors.length === 3;
}

/**
 * Return shard part of selectors
 *
 * @param {Selectors} selectors
 * @return shard selectors only
 */
export function onlyShardKey(selectors) {
  const shardSelectors = selectors.getSelectors()
    .filter((s) => SHARD_KEYS.indexOf(s.getName()) >= 0);

  return new Selectors(selectors.getNameSelector(), shardSelectors);
}

/**
 * Return inside shard part of selectors
 *
 * @param {Selectors} selectors
 * @return new selectors without shard selectors
 */
export function withoutShardKey(selectors) {
  const shardSelectors = selectors.getSelectors()
    .filter((s) => SHARD_KEYS.indexOf(s.getName()) < 0);

  return new Selectors(selectors.getNameSelector(), shardSelectors);
}
