class StringInterpolator {
  /**
   * @param {string} pattern
   * @param {object} params
   * @param {boolean} fillEmpty
   * @return {string} interpolated string
   */
  static interpolatePattern(pattern, params, fillEmpty = false, transformFunc = null) {
    const interpolatedString = StringInterpolator.parse(pattern);
    return StringInterpolator.partialEval(interpolatedString, params, fillEmpty, transformFunc);
  }

  static partialEval(interpolatedString, params, fillEmpty, transformFunc) {
    let result = interpolatedString.literals[0];

    for (let i = 0; i < interpolatedString.variables.length; ++i) {
      const varName = interpolatedString.variables[i];
      let value = params[varName] || '';
      if (transformFunc) {
        value = transformFunc(value);
      }
      const nextLiteral = interpolatedString.literals[i + 1];
      if (value) {
        result += value + nextLiteral;
      } else if (fillEmpty) {
        result += nextLiteral;
      } else {
        result += `{{${varName}}}${nextLiteral}`;
      }
    }
    return result;
  }

  static parse(pattern) {
    const literals = [];
    const variables = [];

    let pos = 0;
    for (;;) {
      const nextStart = pattern.indexOf('{{', pos);
      if (nextStart < 0) {
        literals.push(pattern.substring(pos));
        break;
      }

      const nextEnd = pattern.indexOf('}}', nextStart);
      if (nextEnd < 0) {
        throw new Error(`wrong pattern: ${pattern}`);
      }

      literals.push(pattern.substring(pos, nextStart));
      const variable = pattern.substring(nextStart + '{{'.length, nextEnd).trim();
      if (!variable) {
        throw new Error('empty variable');
      }
      variables.push(variable);
      pos = nextEnd + '}}'.length;
    }

    return { literals, variables };
  }
}

export default StringInterpolator;
