/* eslint-disable no-param-reassign,no-plusplus,max-len,spaced-comment */
import has from 'lodash/has';
import size from 'lodash/size';
import values from 'lodash/values';
import entries from 'lodash/entries';
import uniqWith from 'lodash/uniqWith';
import isEqual from 'lodash/isEqual';

import compareTextWithNumber from '../sortTextWithNumber';
import GradientColor from './GradientColor';
import ColorSchemeType from './ColorSchemeType';
import LineColor from './LineColor';

function putIfAbsent(map, key, value) {
  if (!has(map, key)) {
    map[key] = value;
  }
}

class ColorScheme {
  static EMPTY = new ColorScheme({});

  static UNIQUE_GRADIENTED_LABELS_PATTERN = /^(|.*\D)(\d+)(\D*)$/;

  constructor(colorMap) {
    this.colorMap = colorMap;
  }

  getColorOrAutoColor(label) {
    if (has(this.colorMap, label)) {
      return this.colorMap[label];
    }

    return LineColor.autoColor();
  }

  static construct(colorSchemeParams, labelValues) {
    switch (colorSchemeParams.type) {
      case ColorSchemeType.GRADIENT:
        return ColorScheme.gradientScheme(colorSchemeParams, labelValues);
      case ColorSchemeType.DEFAULT:
        return ColorScheme.EMPTY;
      case ColorSchemeType.AUTO:
        return ColorScheme.autoScheme(colorSchemeParams, labelValues);
      default:
        throw new Error(`unknown color scheme type: ${colorSchemeParams.type}`);
    }
  }

  static autoScheme(colorSchemeParams, labelValues) {
    if (ColorScheme.isUniqueGradientedLabels(labelValues)) {
      return ColorScheme.gradientScheme(colorSchemeParams, labelValues);
    }
    return ColorScheme.EMPTY;
  }

  static constructColorMap(colorIndexes, labels) {
    const colorMap = {};

    if (size(colorIndexes) > 0) {
      const entryList = entries(colorIndexes);
      entryList.sort((a, b) => a[1] - b[1]);

      let currentColorIndex = 0;
      for (let i = 0; i < labels.length; ++i) {
        while (currentColorIndex < entryList.length && i >= entryList[currentColorIndex][1]) {
          ++currentColorIndex;
        }

        let gradientPosition;
        let zone;

        if (currentColorIndex < entryList.length && currentColorIndex > 0) {
          const currentColor = GradientColor.getProperty(entryList[currentColorIndex][0]);
          const currentColorNumber = entryList[currentColorIndex][1];

          const previousColor = GradientColor.getProperty(entryList[currentColorIndex - 1][0]);
          const previousColorNumber = entryList[currentColorIndex - 1][1];

          gradientPosition = (i - previousColorNumber) / (currentColorNumber - previousColorNumber);

          if (previousColor.ordinal < currentColor.ordinal) {
            zone = previousColor.getGradientZone();
          } else {
            zone = currentColor.getGradientZone();
            gradientPosition = 1 - gradientPosition;
          }
        } else if (currentColorIndex >= entryList.length) {
          const currentColor = GradientColor.getProperty(entryList[entryList.length - 1][0]);
          zone = currentColor.getGradientZone();
          gradientPosition = currentColor.getGradientPosition();
        } else {
          const currentColor = GradientColor.getProperty(entryList[0][0]);
          zone = currentColor.getGradientZone();
          gradientPosition = currentColor.getGradientPosition();
        }

        colorMap[labels[i]] = LineColor.gradientColor(zone, gradientPosition);
      }
    }

    return colorMap;
  }

  static gradientScheme(colorSchemeParams, labelValues) {
    const colorLabels = ColorScheme.getGradientColorLabels(colorSchemeParams);
    const uniqueValues = ColorScheme.getUniqueGradientedLabels(labelValues, colorLabels);
    const colorsIndexes = ColorScheme.getGradientColorsPositions(colorLabels, uniqueValues);
    const colorMap = ColorScheme.constructColorMap(colorsIndexes, uniqueValues);
    return new ColorScheme(colorMap);
  }

  static getGradientColorsPositions(colorLabels, uniqueValues) {
    const colorIndexes = ColorScheme.getInputSetGradientColorMap(colorLabels, uniqueValues);
    return ColorScheme.addMissingPointsToGradientMap(colorIndexes, uniqueValues.length);
  }

  static getInputSetGradientColorMap(colorLabels, uniqueValues) {
    const colorNumbers = {};

    for (let i = 0; i < uniqueValues.length; ++i) {
      const value = uniqueValues[i];
      const filteredEntries = entries(colorLabels).filter((e) => e[1] === value);

      if (filteredEntries.length > 0) {
        const foundedColor = filteredEntries[0][0];
        colorNumbers[foundedColor] = i;
      }
    }

    return colorNumbers;
  }

  static isUniqueGradientedLabels(labels) {
    labels = labels.filter((label) => label !== 'inf');

    if (labels.length === 0) {
      return false;
    }

    const firstLabel = labels[0];

    const firstLabelPrefixOrSuffix = ColorScheme.labelPrefixAndSuffixOrNull(firstLabel);
    if (firstLabelPrefixOrSuffix === null) {
      return false;
    }

    for (let i = 1; i < labels.length; ++i) {
      const currentLabel = labels[i];
      const currentLabelPrefixAndSuffix = ColorScheme.labelPrefixAndSuffixOrNull(currentLabel);
      if (!isEqual(firstLabelPrefixOrSuffix, currentLabelPrefixAndSuffix)) {
        return false;
      }
    }
    return true;
  }

  static labelPrefixAndSuffixOrNull(label) {
    const matcher = ColorScheme.UNIQUE_GRADIENTED_LABELS_PATTERN.exec(label);

    if (matcher !== null && matcher.length === 4 && ColorScheme.tryParseDecimalLong(matcher[2])) {
      return [matcher[1], matcher[3]];
    }

    return null;
  }

  static tryParseDecimalLong(s) {
    const num = Number(s);
    return !isNaN(num);
  }

  static getUniqueGradientedLabels(labels, colorLabels) {
    const strings = [];
    strings.push(...labels);
    strings.push(...values(colorLabels).filter((s) => !!s));
    strings.sort(compareTextWithNumber);

    return uniqWith(strings, (a, b) => a === b);
  }

  static getGradientColorLabels(colorSchemeParams) {
    const colorLabels = {};
    ColorScheme.putColorLabelIfPresent(colorLabels, GradientColor.GREEN, colorSchemeParams.green);
    ColorScheme.putColorLabelIfPresent(colorLabels, GradientColor.YELLOW, colorSchemeParams.yellow);
    ColorScheme.putColorLabelIfPresent(colorLabels, GradientColor.RED, colorSchemeParams.red);
    ColorScheme.putColorLabelIfPresent(colorLabels, GradientColor.VIOLET, colorSchemeParams.violet);
    return colorLabels;
  }

  static putColorLabelIfPresent(colorLabels, color, label) {
    if (label) {
      colorLabels[color] = label;
    }
  }

  static addMissingPointsToGradientMap(colorNumbers, setSize) {
    const mapOrder = ColorScheme.getMapOrder(colorNumbers);
    const allKeys = GradientColor.values();
    const firstKeyOrdinal = allKeys[0].getOrdinal();
    const lastKeyOrdinal = allKeys[allKeys.length - 1].getOrdinal();
    if (mapOrder > 0) {
      putIfAbsent(colorNumbers, firstKeyOrdinal, setSize);
      putIfAbsent(colorNumbers, lastKeyOrdinal, 0);
    } else {
      putIfAbsent(colorNumbers, firstKeyOrdinal, 0);
      putIfAbsent(colorNumbers, lastKeyOrdinal, setSize);
    }

    const keys = GradientColor.values().filter((color) => has(colorNumbers, color.getOrdinal()));

    const newColorNumber = {};
    let i = 0;
    let j = 0;
    while (i < allKeys.length && j < keys.length) {
      const presentKey = keys[j];
      const key = allKeys[i];

      const order = presentKey.getOrdinal() - key.getOrdinal();
      if (order > 0) {
        if (j > 0) {
          const previousPresentKey = keys[j - 1];
          const pKeyOrdinal = previousPresentKey.getOrdinal();
          const keyOrdinal = presentKey.getOrdinal();
          const absentKeyOrdinal = key.getOrdinal();
          const pKeyNumber = colorNumbers[pKeyOrdinal];
          const keyNumber = colorNumbers[keyOrdinal];
          const absentKeyNumber = Math.trunc(
            (((keyNumber - pKeyNumber) * (absentKeyOrdinal - pKeyOrdinal))
            / (keyOrdinal - pKeyOrdinal)) + pKeyNumber,
          );
          newColorNumber[key.getOrdinal()] = absentKeyNumber;
        }
        i++;
      } else if (order === 0) {
        newColorNumber[presentKey.getOrdinal()] = colorNumbers[presentKey.getOrdinal()];
        i++;
        j++;
      } else {
        throw new Error('we must not reach there');
      }
    }
    return newColorNumber;
  }

  static getMapOrder(colorNumbers) {
    if (size(colorNumbers) > 1) {
      const initialKeys = GradientColor.values().filter((color) => has(colorNumbers, color.getOrdinal()));

      const firstColor = colorNumbers[initialKeys[0].getOrdinal()];
      const lastColor = colorNumbers[initialKeys[initialKeys.length - 1].getOrdinal()];
      return firstColor - lastColor;
    }

    return 0;
  }
}

export default ColorScheme;
