/* eslint-disable import/no-cycle */
import * as CssColorName from './CssColorName';

function isHexDigit(c) {
  return (c >= '0' && c <= '9') || (c >= 'a' && c <= 'f') || (c >= 'A' && c <= 'F');
}

function isHexColor(color) {
  if (!color.startsWith('#')) {
    return false;
  }
  if (color.length !== 4 && color.length !== 7) {
    return false;
  }
  for (let i = 1; i < color.length; ++i) {
    if (!isHexDigit(color[i])) {
      return false;
    }
  }
  return true;
}

function parseHex6Component(component) {
  if (component.length !== 2) {
    throw new Error(`unknown hex component: ${component}`);
  }

  return parseInt(component, 16);
}

function parseRgbComponent(component) {
  const value = parseInt(component, 10);
  if (!(value >= 0 && value <= 255)) {
    throw new Error(`unknown rgb component: ${component}`);
  }
  return value;
}

function parseRgbDeclaration(rgbDeclaration) {
  let rgbParams;

  if (rgbDeclaration.startsWith('rgb(') && rgbDeclaration.endsWith(')')) {
    rgbParams = rgbDeclaration.substr(0, rgbDeclaration.length - 1).substr(4).trim().split(/\s*,\s*/);
  } else {
    throw new Error(`invalid rgb declaration: ${rgbDeclaration}`);
  }

  if (rgbParams.length !== 3) {
    throw new new Error(`failed to parse rgb params in "${rgbDeclaration}": ${rgbParams}`)();
  }

  return {
    r: parseRgbComponent(rgbParams[0]),
    g: parseRgbComponent(rgbParams[1]),
    b: parseRgbComponent(rgbParams[2]),
  };
}

function isRgbDeclaration(rgbDeclaration) {
  try {
    parseRgbDeclaration(rgbDeclaration);
    return true;
  } catch (e) {
    return false;
  }
}

export function parseHex(hex) {
  if (!hex.startsWith('#')) {
    throw new Error(`failed to parse hex: ${hex}`);
  }

  let rgb = hex.substring(1);
  if (rgb.length === 3) {
    rgb = rgb.replace(/(\w)/g, '$1$1');
  } else if (rgb.length !== 6) {
    throw new Error(`invalid rgb declaration length: ${hex}`);
  }

  return {
    r: parseHex6Component(rgb.substring(0, 2)),
    g: parseHex6Component(rgb.substring(2, 4)),
    b: parseHex6Component(rgb.substring(4, 6)),
  };
}

export function isValid(color) {
  return CssColorName.isValidName(color)
    || isHexColor(color) || isRgbDeclaration(color);
}

export function parseColorOrNull(color) {
  try {
    return CssColorName.parseColorName(color);
  } catch (e) {
    // It's not color name
  }

  try {
    return parseHex(color);
  } catch (e) {
    // It's not color in hex format
  }

  try {
    return parseRgbDeclaration(color);
  } catch (e) {
    // It's not color in rgb format, return null
  }

  return null;
}
