/* eslint-disable no-mixed-operators,no-multi-assign,no-param-reassign */
import ColorScheme from './ColorScheme';
import LineColor from './LineColor';
import GradientZone from './GradientZone';

const STARTING_HUE_POINTS = [80, 200, 50, 0, 160, 250];
const REGION_HUE_POINTS = [90, 200, 50, 10, 160, 330];

function hue2rgb(p, q, t) {
  if (t < 0) {
    t += 1;
  }
  if (t > 1) {
    t -= 1;
  }
  if (t < 1 / 6) {
    return p + (q - p) * 6 * t;
  }
  if (t < 1 / 2) {
    return q;
  }
  if (t < 2 / 3) {
    return p + (q - p) * (2 / 3 - t) * 6;
  }
  return p;
}

function fixAttr(value) {
  if (value > 255) {
    return 255;
  }
  if (value < 0) {
    return 0;
  }

  return value.toFixed();
}

export function formatRgb(rgb) {
  return `rgb(${fixAttr(rgb.r)},${fixAttr(rgb.g)},${fixAttr(rgb.b)})`;
}

export function convertRgbToHsl(r, g, b) {
  const rr = r / 255;
  const gg = g / 255;
  const bb = b / 255;

  const cMax = Math.max(rr, Math.max(gg, bb));
  const cMin = Math.min(rr, Math.min(gg, bb));
  const delta = cMax - cMin;

  let h;

  if (delta === 0) {
    h = 0;
  } else if (cMax === bb) {
    h = ((rr - gg) / delta + 4);
  } else if (cMax === gg) {
    h = ((bb - rr) / delta + 2);
  } else {
    h = ((gg - bb) / delta % 6);
  }
  h /= 6;

  const l = (cMax + cMin) / 2;
  const s = delta === 0 ? 0 : delta / (1 - Math.abs(2 * l - 1));

  return { h, s, l };
}

export function convertHslToRgb(h, s, l) {
  let r;
  let g;
  let b;

  if (s === 0) {
    r = g = b = l; // achromatic
  } else {
    const q = l < 0.5 ? l * (1 + s) : l + s - l * s;
    const p = 2 * l - q;
    r = hue2rgb(p, q, h + 1 / 3);
    g = hue2rgb(p, q, h);
    b = hue2rgb(p, q, h - 1 / 3);
  }

  return { r: r * 255, g: g * 255, b: b * 255 };
}

export function getColorAlexPetrov(n) {
  const regionSize = 40;
  let hue;
  let saturation = 100;
  let lightness = 43;
  const magic = 7;

  if (n < STARTING_HUE_POINTS.length) {
    hue = STARTING_HUE_POINTS[n];
  } else {
    const region = n % REGION_HUE_POINTS.length;
    const iteration = Math.floor(n / REGION_HUE_POINTS.length);

    const hueShift = (magic * iteration) % regionSize;

    hue = (REGION_HUE_POINTS[region] + hueShift) % 360;

    // Неразличимый зеленый
    if (hue > 120 && hue < 150) {
      hue = (hue + 50) % 360;
    }
    // Фиолетово-розовый
    if (hue > 270 && hue < 330) {
      hue = (hue + 70) % 360;
    }
    // Неразличимый синий
    if (hue > 220 && hue < 250) {
      hue = (hue + 40) % 360;
    }

    const saturationShift = (13 * (iteration + region)) % 20;
    saturation = 70 + saturationShift;

    const lightnessShift = (11 * (iteration + region)) % 25;
    if ((region + iteration + 1) % 2 > 0) {
      lightness = 45 - lightnessShift;
    } else {
      lightness = 45 + lightnessShift;
    }
  }

  const newRgb = convertHslToRgb(hue / 360.0, saturation / 100.0, lightness / 100.0);

  return formatRgb(newRgb);
}

function lin(pos, rfrom, rto) {
  const r = Math.floor((pos * (rto - rfrom)) + rfrom);

  const rmin = Math.min(rfrom, rto);
  const rmax = Math.max(rfrom, rto);

  if (r < rmin) {
    return rmin;
  }

  if (r > rmax) {
    return rmax;
  }

  return r;
}

function lina(pos, rfrom, rto) {
  const r = lin(pos, rfrom.r, rto.r);
  const g = lin(pos, rfrom.g, rto.g);
  const b = lin(pos, rfrom.b, rto.b);

  return { r, g, b };
}

export function getGradientColor(pos, begin, end) {
  const rgb = lina(pos, begin, end);
  return formatRgb(rgb);
}

export function getGradientColorByZone(pos, zone) {
  const rgb = lina(pos, zone.begin, zone.end);
  return formatRgb(rgb);
}

export function getAlexPetrovColorsForN(n) {
  const result = [];
  for (let i = 0; i < n; ++i) {
    result[i] = getColorAlexPetrov(i);
  }
  return result;
}

function getColorsForNamesImpl(names, colorSchemeParams) {
  const colorScheme = ColorScheme.construct(colorSchemeParams, names);

  return names.map((name, index) => {
    const lineColor = colorScheme.getColorOrAutoColor(name);

    if (lineColor.type === LineColor.Type.GRADIENT) {
      const zone = GradientZone.values[lineColor.zone];
      return getGradientColor(lineColor.position, zone.begin, zone.end);
    }

    return getColorAlexPetrov(index);
  });
}

export function getColorsForNames(names, colorSchemeParams) {
  if (!colorSchemeParams) {
    return getAlexPetrovColorsForN(names.length);
  }

  try {
    return getColorsForNamesImpl(names, colorSchemeParams);
  } catch (e) {
    console.error('failed to get colors for lines', e);
    return getAlexPetrovColorsForN(names.length);
  }
}
