import entries from 'lodash/entries';
import flatMap from 'lodash/flatMap';

import { getColorsForNames } from './color/colors';
import sortTextWithNumber from './sortTextWithNumber';
import { doubleQuote } from './Quoter';
import Selectors from './Selectors';

export const LINE_NAMING_MODE = {
  DEFAULT: 'DEFAULT',
  ALERTING_SPECIALLY: 'ALERTING_SPECIALLY',
  OLD_UI: 'OLD_UI',
};

export const COLUMN_TYPE = {
  SIMPLE: 0,
  ALIAS: 1,
  METRIC_NAME: 2,
  METRIC_LABEL: 3,
  PROPERTY: 4,
};

function formatLabels(nonUniqueLabels, line) {
  const formattedLabels = nonUniqueLabels.map((labelKey) => {
    let formattedLabelKey = labelKey;
    if (formattedLabelKey.indexOf('-') >= 0) {
      formattedLabelKey = `"${labelKey}"`;
    }
    const labelValue = line.rawSeries.labels[labelKey] || '-';

    return `${formattedLabelKey}=${doubleQuote(labelValue)}`;
  })
    .join(', ');

  return `{${formattedLabels}}`;
}

function formatProperties(line, nonUniqueProps) {
  let formattedProps = nonUniqueProps
    .map((prop) => `${prop}=${doubleQuote(line.properties[prop] || '-')}`)
    .join(', ');
  if (formattedProps) {
    formattedProps = `{${formattedProps}}`;
  }
  return formattedProps;
}

function createLegendNamesForLineInDefaultMode(
  line,
  isUniqueAlias,
  isUniqueName,
  nonUniqueLabels,
  nonUniqueProps,
) {
  const legendNames = [];
  if (!isUniqueAlias) {
    legendNames.push(line.rawSeries.alias || '-');
  }
  if (!isUniqueName) {
    legendNames.push(line.rawSeries.name || '-');
  }
  legendNames.push(...nonUniqueLabels.map((label) => line.rawSeries.labels[label] || '-'));
  legendNames.push(...nonUniqueProps.map((prop) => line.properties[prop] || '-'));
  return legendNames;
}

function createNameForLineInOldFormat(
  line,
  isUniqueName,
  nonUniqueLabels,
  nonUniqueProps,
) {
  const nameParts = [];

  if (!isUniqueName) {
    if (nonUniqueLabels.length === 0 && nonUniqueProps.length === 0) {
      return line.rawSeries.name || '-';
    }
  }

  if (nonUniqueLabels.length > 0) {
    if (nonUniqueLabels.length === 1 && nonUniqueProps.length === 0) {
      const label = nonUniqueLabels[0];
      return line.rawSeries.labels[label] || '-';
    }

    nameParts.push(...nonUniqueLabels.map((labelKey) => line.rawSeries.labels[labelKey] || '-'));

    nameParts.push(...nonUniqueProps
      .map((prop) => line.properties[prop] || '-'));

    return nameParts.join(', ');
  }

  if (nonUniqueProps.length > 0) {
    if (nonUniqueProps.length === 1) {
      const prop = nonUniqueProps[0];
      return line.properties[prop] || '-';
    }

    return formatProperties(line, nonUniqueProps);
  }

  return '-';
}

function createNameForLineInDefaultMode(
  line,
  isUniqueName,
  nonUniqueLabels,
  nonUniqueProps,
  oldFormat,
) {
  // If alias is present - show alias only
  if (line.rawSeries.alias) {
    return line.rawSeries.alias || '-';
  }

  if (oldFormat) {
    return createNameForLineInOldFormat(
      line,
      isUniqueName,
      nonUniqueLabels,
      nonUniqueProps,
      oldFormat,
    );
  }

  if (!isUniqueName) {
    if (nonUniqueLabels.length === 0 && nonUniqueProps.length === 0) {
      return line.rawSeries.name || '-';
    }

    const formattedName = Selectors._formatNameSelector(line.rawSeries.name);
    const formattedLabels = formatLabels(nonUniqueLabels, line, oldFormat);
    let formattedProps = formatProperties(line, nonUniqueProps);
    if (formattedProps) {
      formattedProps = `${formattedProps}`;
    }
    return `${formattedName}${formattedLabels}${formattedProps}`;
  }

  if (nonUniqueLabels.length > 0) {
    if (nonUniqueLabels.length === 1 && nonUniqueProps.length === 0) {
      const label = nonUniqueLabels[0];
      return line.rawSeries.labels[label] || '-';
    }

    const formattedLabels = formatLabels(nonUniqueLabels, line, oldFormat);
    let formattedProps = formatProperties(line, nonUniqueProps);
    if (formattedProps) {
      formattedProps = `${formattedProps}`;
    }
    return `${formattedLabels}${formattedProps}`;
  }

  if (nonUniqueProps.length > 0) {
    if (nonUniqueProps.length === 1) {
      const prop = nonUniqueProps[0];
      return line.properties[prop] || '-';
    }

    return formatProperties(line, nonUniqueProps);
  }

  return '-';
}

function isUniqueProperty(lines, mapper) {
  if (lines.length <= 1) {
    return true;
  }

  const firstValue = mapper(lines[0]);

  for (let i = 1; i < lines.length; ++i) {
    const value = mapper(lines[i]);
    if (firstValue !== value) {
      return false;
    }
  }

  return true;
}

export function findNonUniqueKeys(lines, mapper) {
  if (lines.length <= 1) {
    return [];
  }

  const firstObj = { ...mapper(lines[0]) };

  const nonUniqueKeys = new Set();

  for (let i = 1; i < lines.length; ++i) {
    const obj = mapper(lines[i]);

    Object.entries(obj).forEach((entry) => {
      const key = entry[0];
      const value = entry[1];
      const firstValue = firstObj[key];
      if (firstValue !== value) {
        nonUniqueKeys.add(key);
      }
    });

    Object.entries(firstObj).forEach((entry) => {
      const key = entry[0];
      const firstValue = entry[1];
      const value = obj[key];
      if (firstValue !== value) {
        nonUniqueKeys.add(key);
      }
    });
  }

  return Array.from(nonUniqueKeys).sort();
}

function makeNamesUnique(lines) {
  const uniqueLabels = {};
  const duplicatedLabelsWithCounters = {};
  lines.map((line) => line.name).forEach((name) => {
    if (name in uniqueLabels) {
      duplicatedLabelsWithCounters[name] = 0;
    } else {
      uniqueLabels[name] = true;
    }
  });

  return lines.map((line) => {
    const { name } = line;
    if (name in duplicatedLabelsWithCounters) {
      let copyNumber = duplicatedLabelsWithCounters[name];
      copyNumber += 1;
      duplicatedLabelsWithCounters[name] = copyNumber;
      const newName = `${name} (${copyNumber})`;
      return { ...line, name: newName };
    }
    return line;
  });
}

function isAliasedLines(lines) {
  return lines.map((line) => line.rawSeries.alias).every((alias) => !!alias);
}

function createNamesForAliasedLines(lines) {
  let namedLines = lines.map((line) => ({
    ...line,
    name: line.rawSeries.alias,
  }));

  namedLines = makeNamesUnique(namedLines);

  namedLines.forEach((line) => {
    // eslint-disable-next-line no-param-reassign
    line.legendNames = [line.name];
  });

  const columns = [{
    title: 'Alias',
    type: COLUMN_TYPE.ALIAS,
  }];

  return { namedLines, columns };
}

function createNamesForAllDuplicatedLines(lines) {
  const namedLines = lines.map((line, index) => {
    const lineName = `Line (${index})`;
    return ({ ...line, name: lineName, legendNames: [lineName] });
  });
  const columns = [{ title: 'alias', type: COLUMN_TYPE.SIMPLE }];
  return { namedLines, columns };
}

function createNamesUsingDefaultModeImpl(lines, oldFormat = false) {
  const isUniqueAlias = isUniqueProperty(lines, (line) => line.rawSeries.alias);
  const isUniqueName = isUniqueProperty(lines, (line) => line.rawSeries.name);
  const nonUniqueLabels = findNonUniqueKeys(lines, (line) => line.rawSeries.labels);
  const nonUniqueProps = findNonUniqueKeys(lines, (line) => line.properties);

  if (isUniqueAlias && isUniqueName
    && nonUniqueLabels.length === 0
    && nonUniqueProps.length === 0) {
    return createNamesForAllDuplicatedLines(lines);
  }

  let namedLines = lines.map((line) => {
    const name = createNameForLineInDefaultMode(
      line,
      isUniqueName,
      nonUniqueLabels,
      nonUniqueProps,
      oldFormat,
    );

    const legendNames = createLegendNamesForLineInDefaultMode(
      line,
      isUniqueAlias,
      isUniqueName,
      nonUniqueLabels,
      nonUniqueProps,
    );

    return ({ ...line, name, legendNames });
  });

  namedLines = makeNamesUnique(namedLines);

  const columns = [];
  if (!isUniqueAlias) {
    columns.push({ title: 'Alias', type: COLUMN_TYPE.ALIAS });
  }
  if (!isUniqueName) {
    columns.push({ title: 'Name', type: COLUMN_TYPE.METRIC_NAME });
  }
  // eslint-disable-next-line max-len
  columns.push(...nonUniqueLabels.map((label) => ({ title: label, type: COLUMN_TYPE.METRIC_LABEL })));
  columns.push(...nonUniqueProps.map((prop) => ({ title: prop, type: COLUMN_TYPE.PROPERTY })));

  return { namedLines, columns };
}

function createNameForSingleLineUsingDefaultMode(lines) {
  const line = lines[0];
  const name = 'Single line';
  const namedLines = [{ ...line, name, legendNames: [name] }];
  const columns = [{
    title: 'Name',
    type: COLUMN_TYPE.SIMPLE,
  }];
  return { namedLines, columns };
}

function createNameForSingleLineUsingOldMode(lines) {
  const line = lines[0];

  const labels = line.rawSeries.labels || {};

  let lineName;
  let columnName;
  if (labels.sensor) {
    lineName = labels.sensor;
    columnName = 'sensor';
  } else if (labels.path) {
    lineName = labels.path;
    columnName = 'path';
  } else if (labels.metric) {
    lineName = labels.metric;
    columnName = 'metric';
  } else if (labels.signal) {
    lineName = labels.signal;
    columnName = 'signal';
  } else if (labels.name) {
    lineName = labels.name;
    columnName = 'name';
  } else {
    lineName = 'Line';
    columnName = 'Name';
  }

  const columnType = columnName === 'Name' ? COLUMN_TYPE.SIMPLE : COLUMN_TYPE.METRIC_LABEL;

  const namedLines = [{ ...line, name: lineName, legendNames: [lineName] }];
  const columns = [{
    title: columnName,
    type: columnType,
  }];
  return { namedLines, columns };
}

function createNamesUsingDefaultMode(lines, oldFormat = false) {
  if (isAliasedLines(lines)) {
    return createNamesForAliasedLines(lines);
  }

  if (lines.length === 1) {
    if (oldFormat) {
      return createNameForSingleLineUsingOldMode(lines);
    }
    return createNameForSingleLineUsingDefaultMode(lines);
  }

  return createNamesUsingDefaultModeImpl(lines, oldFormat);
}

function createNamesForAlertingSpecially(lines) {
  const seriesByAlias = {};

  lines.forEach((s) => {
    const alias = s.rawSeries.alias || '';
    if (!seriesByAlias[alias]) {
      seriesByAlias[alias] = [];
    }
    seriesByAlias[alias].push(s);
  });

  const aliasEntries = entries(seriesByAlias);

  if (aliasEntries.length === 1) {
    return createNamesUsingDefaultMode(lines);
  }

  const namedLines = flatMap(aliasEntries, ([alias, aliasLines]) => {
    const nonUniqueLabels = findNonUniqueKeys(aliasLines, (line) => line.rawSeries.labels);

    if (nonUniqueLabels.length === 0) {
      return aliasLines.map((line) => ({ ...line, name: alias, legendNames: [alias] }));
    }

    return aliasLines.map((line) => {
      const formattedLabels = formatLabels(nonUniqueLabels, line);
      const name = `${alias} ${formattedLabels}`;
      return { ...line, name, legendNames: [name] };
    });
  });

  return {
    namedLines,
    columns: [{ title: 'Name', type: COLUMN_TYPE.SIMPLE }],
  };
}

function createNamesForLines(lines, lineNamingMode = '') {
  if (lines.length === 0) {
    return { namedLines: [], columns: [] };
  }

  switch (lineNamingMode) {
    case LINE_NAMING_MODE.ALERTING_SPECIALLY:
      return createNamesForAlertingSpecially(lines);
    case LINE_NAMING_MODE.OLD_UI:
      return createNamesUsingDefaultMode(lines, true);
    default:
      return createNamesUsingDefaultMode(lines, false);
  }
}

function mapExplanationToHighchartSeries(line, index) {
  const { name } = line;
  const { color } = line;
  const data = Array(line.values.length);

  for (let i = 0; i < line.values.length; ++i) {
    let value = line.values[i];

    if (isNaN(value) || !Number.isFinite(value)) {
      value = NaN;
    }

    data[i] = [line.timestamps[i], value];
  }

  return {
    name, color, data, index,
  };
}

function getValuesForProperty(series, prop) {
  return series.values.map((value) => {
    if (!value) {
      return NaN;
    }

    const propValue = value[prop];
    if (isNaN(propValue)) {
      return NaN;
    }

    return propValue;
  });
}

function createPropertyLineFromSeries(series, prop) {
  const values = getValuesForProperty(series, prop);

  return {
    rawSeries: series,
    properties: { summary: prop },
    timestamps: series.timestamps || [],
    values,
    refIndex: series.refIndex || 0,
  };
}

function getAllBoundsFromHistograms(series) {
  const bounds = new Set();

  series.values.forEach((value) => {
    if (value) {
      value.bounds.forEach((bound) => bounds.add(bound));
    }
  });

  if (bounds.size === 0) {
    return [];
  }

  const boundsArray = Array.from(bounds);

  boundsArray.sort();

  return [...boundsArray, Infinity];
}

function getValuesForBound(series, bound) {
  return series.values.map((value) => {
    if (!value) {
      return NaN;
    }

    if (bound === Infinity) {
      const { inf } = value;
      return isNaN(inf) ? NaN : inf;
    }

    const boundIndex = value.bounds.indexOf(bound);

    if (boundIndex >= 0) {
      return value.buckets[boundIndex];
    }

    return NaN;
  });
}

function createBoundLineFromSeries(series, bound) {
  const values = getValuesForBound(series, bound);
  const boundStr = bound === Infinity ? 'inf' : String(bound);

  return {
    rawSeries: series,
    properties: { bound: boundStr },
    values,
    timestamps: series.timestamps || [],
    refIndex: series.refIndex || 0,
  };
}

function createSimpleLineFromSeries(series) {
  return {
    rawSeries: series,
    values: series.values,
    timestamps: series.timestamps || [],
    properties: {},
    refIndex: series.refIndex || 0,
  };
}

function flatSeriesForHistogram(series) {
  const bounds = getAllBoundsFromHistograms(series);
  return bounds.map((bound) => createBoundLineFromSeries(series, bound));
}

function formatLogHistogramBound(bucketIndex, startPower, base) {
  return `${base}^${startPower + bucketIndex}`;
}

function getAllBoundsFromLogHistograms(series) {
  const bounds = new Set();

  series.values.forEach((value) => {
    if (value) {
      for (let i = 0; i < value.buckets.length; ++i) {
        const boundName = formatLogHistogramBound(i, value.startPower, value.base);
        bounds.add(boundName);
      }
    }
  });

  if (bounds.size === 0) {
    return [];
  }
  const boundsArray = Array.from(bounds);
  boundsArray.sort();
  return [...boundsArray];
}

function createCountZeroLineFromSeries(series) {
  const values = series.values.map((value) => (!value ? NaN : value.countZero));

  return {
    rawSeries: series,
    values,
    timestamps: series.timestamps || [],
    properties: { bound: '0' },
    refIndex: series.refIndex || 0,
  };
}

function createLogBoundLineFromSeries(series, bound) {
  const values = series.values.map((value) => {
    if (!value) {
      return NaN;
    }

    for (let boundIndex = 0; boundIndex < value.buckets.length; ++boundIndex) {
      const curBound = formatLogHistogramBound(boundIndex, value.startPower, value.base);
      if (curBound === bound) {
        return value.buckets[boundIndex];
      }
    }

    return NaN;
  });

  return {
    rawSeries: series,
    values,
    timestamps: series.timestamps || [],
    properties: { bound },
    refIndex: series.refIndex || 0,
  };
}

function flatSeriesForLogHistogram(series) {
  const bounds = getAllBoundsFromLogHistograms(series);
  const countZeroLine = createCountZeroLineFromSeries(series);
  const boundLines = bounds.map((bound) => createLogBoundLineFromSeries(series, bound));
  return [countZeroLine, ...boundLines];
}

function flatSeriesForSummary(series) {
  return ['count', 'min', 'max', 'sum', 'last']
    .map((prop) => createPropertyLineFromSeries(series, prop));
}

function flatSeriesToLines(series) {
  if (series.kind === 'HIST' || series.kind === 'HIST_RATE') {
    return flatSeriesForHistogram(series);
  }

  if (series.kind === 'LOG_HISTOGRAM') {
    return flatSeriesForLogHistogram(series);
  }

  if (series.kind === 'DSUMMARY' || series.kind === 'ISUMMARY') {
    return flatSeriesForSummary(series);
  }

  return [createSimpleLineFromSeries(series)];
}

function flatAllSeriesToLines(series) {
  return series.map(flatSeriesToLines).reduce((a, b) => a.concat(b));
}

export function fixSeries(series, params = {}) {
  if (!series || series.length === 0) {
    return { lines: [], highchartsLines: [], columns: [] };
  }

  let lines = flatAllSeriesToLines(series);

  const namedLinesAndColumns = createNamesForLines(lines, params.mode);

  lines = namedLinesAndColumns.namedLines;
  const { columns } = namedLinesAndColumns;

  lines.sort((a, b) => {
    if (a.refIndex === b.refIndex) {
      return sortTextWithNumber(a.name, b.name);
    }

    return a.refIndex - b.refIndex;
  });

  const names = lines.map((s) => s.name);
  const colors = getColorsForNames(names, params.colorScheme);
  lines = lines.map((line, index) => ({ ...line, color: colors[index] || '#ddd' }));

  const highchartsLines = lines.map(mapExplanationToHighchartSeries);

  return { lines, highchartsLines, columns };
}

export default {};
