const MILLISECONDS_PER_SECOND = 1000;
const MILLISECONDS_PER_MINUTE = MILLISECONDS_PER_SECOND * 60;
const MILLISECONDS_PER_HOUR = MILLISECONDS_PER_MINUTE * 60;
const MILLISECONDS_PER_DAY = MILLISECONDS_PER_HOUR * 24;
const MILLISECONDS_PER_WEEK = MILLISECONDS_PER_DAY * 7;

const DURATION_ELEM_REGEX = /^(\d+)(w|d|h|ms|m|s)/;

export function parseDuration(text, ignoreMs) {
  if (!text) {
    return null;
  }

  if (text === '0') {
    return 0;
  }

  const neg = text.startsWith('-');

  let unparsedText = neg ? text.substring(1) : text;
  let duration = 0;

  while (unparsedText.length > 0) {
    const matchResult = DURATION_ELEM_REGEX.exec(unparsedText);
    if (matchResult == null) {
      return null;
    }

    const count = parseInt(matchResult[1], 10);

    switch (matchResult[2]) {
      case 'w':
        duration += count * MILLISECONDS_PER_WEEK;
        break;
      case 'd':
        duration += count * MILLISECONDS_PER_DAY;
        break;
      case 'h':
        duration += count * MILLISECONDS_PER_HOUR;
        break;
      case 'm':
        duration += count * MILLISECONDS_PER_MINUTE;
        break;
      case 's':
        duration += count * MILLISECONDS_PER_SECOND;
        break;
      case 'ms':
        if (ignoreMs) {
          return null;
        }
        duration += count;
        break;
      default:
        return null;
    }

    unparsedText = unparsedText.substring(matchResult[0].length);
  }

  return neg ? -duration : duration;
}

export function formatDuration(millis) {
  if (millis === 0) {
    return '0';
  }

  if (millis === null || isNaN(millis)) {
    return '';
  }

  const neg = millis < 0;

  let remainderMillis = millis;

  if (neg) {
    remainderMillis = -remainderMillis;
  }

  const days = Math.floor(remainderMillis / MILLISECONDS_PER_DAY);
  remainderMillis -= days * MILLISECONDS_PER_DAY;

  const hours = Math.floor(remainderMillis / MILLISECONDS_PER_HOUR);
  remainderMillis -= hours * MILLISECONDS_PER_HOUR;

  const minutes = Math.floor(remainderMillis / MILLISECONDS_PER_MINUTE);
  remainderMillis -= minutes * MILLISECONDS_PER_MINUTE;

  const seconds = Math.floor(remainderMillis / MILLISECONDS_PER_SECOND);
  remainderMillis -= seconds * MILLISECONDS_PER_SECOND;

  let sb = '';
  if (days !== 0) {
    sb += `${days}d`;
  }
  if (hours !== 0) {
    sb += `${hours}h`;
  }
  if (minutes !== 0) {
    sb += `${minutes}m`;
  }
  if (seconds !== 0) {
    sb += `${seconds}s`;
  }
  if (remainderMillis !== 0) {
    sb += `${remainderMillis}ms`;
  }

  if (sb.length === 0) {
    return '0';
  }

  return `${neg ? '-' : ''}${sb.toString()}`;
}

export function formatDurationSecs(secs) {
  const millis = secs * 1000;
  return formatDuration(millis);
}

export function parseDurationSecs(text) {
  const millis = parseDuration(text, true);
  return millis === null ? null : Math.trunc(millis / 1000);
}
