import values from 'lodash/values';

const ENV_TYPE = {
  DEV: 0,
  TEST: 1,
  PRE: 2,
  PROD: 3,
};

export const ENV_INSTANCE = {
  SERVICE: 0,
  CLOUD: 1,
};

const SOLOMON_INFRA_OPTIONS = {
  DEV: { preset: 'QebaB9kAgDd' },
  PRESTABLE: { preset: 'QebaB9kAgDd' },
  PRODUCTION: { preset: 'RpAKZnb67YE' },
};
const YC_SOLOMON_INFRA_OPTIONS = {
  GPN: { preset: '8qWQrHgwcQr' },
  ISRAEL: { preset: 'W36vxXrA87d' },
  PREPROD: { preset: 'G89XzSKTNvU' },
  PRODUCTION: { preset: 'gDvwYV1WzTZ' },
};

const localHosts = [
  'localhost:3000',
  'localhost.yandex-team.ru:3000',
];

const isLocal = localHosts.indexOf(window.location.host) >= 0;

export const ENVIRONMENTS = {
  DEV: {
    name: 'DEV',
    type: ENV_TYPE.DEV,
    instance: ENV_INSTANCE.SERVICE,
    hosts: [
      'solomon-dev-myt-00.search.yandex.net:3000',
    ],
    dcs: [],
    dataOptions: {
      projectId: 'solomon',
      coreCluster: 'prestable',
      alertingCluster: 'prestable',
      hasProcessingShardData: true,
      hasStorageShardData: true,
    },
    infraOptions: SOLOMON_INFRA_OPTIONS.DEV,
  },
  TEST: {
    name: 'TEST',
    type: ENV_TYPE.TEST,
    instance: ENV_INSTANCE.SERVICE,
    hosts: ['solomon-test.yandex-team.ru'],
    dcs: [{
      id: 'sas',
      title: 'Sas',
      host: 'Sas',
      spCluster: 'testing',
    }],
    dataOptions: {
      projectId: 'solomon',
      coreCluster: 'testing',
      alertingCluster: 'testing',
      hasProcessingShardData: true,
      hasStorageShardData: true,
    },
  },
  PRESTABLE: {
    name: 'PRESTABLE',
    type: ENV_TYPE.PRE,
    instance: ENV_INSTANCE.SERVICE,
    hosts: [
      'solomon-pre.yandex-team.ru',
      'solomon-prestable.yandex-team.ru',
    ],
    dcs: [{
      id: 'vla',
      title: 'Vla',
      host: 'Vla',
      spCluster: 'prestable',
    }],
    dataOptions: {
      projectId: 'solomon',
      coreCluster: 'prestable',
      alertingCluster: 'prestable',
      hasProcessingShardData: true,
      hasStorageShardData: true,
    },
    infraOptions: SOLOMON_INFRA_OPTIONS.PRESTABLE,
  },
  PRODUCTION: {
    name: 'PRODUCTION',
    type: ENV_TYPE.PROD,
    instance: ENV_INSTANCE.SERVICE,
    hosts: [
      'solomon.yandex-team.ru',
      'monitoring.yandex-team.ru',
    ],
    dcs: [
      {
        id: 'sas',
        title: 'Sas',
        host: 'Sas',
        spCluster: 'stockpile_sas',
      },
      {
        id: 'vla',
        title: 'Vla',
        host: 'Vla',
        spCluster: 'stockpile_vla',
      },
    ],
    dataOptions: {
      projectId: 'solomon',
      coreCluster: 'production',
      alertingCluster: 'production',
      hasProcessingShardData: true,
      hasStorageShardData: true,
    },
    infraOptions: SOLOMON_INFRA_OPTIONS.PRODUCTION,
  },
  CLOUD_PREPROD: {
    name: 'CLOUD_PREPROD',
    type: ENV_TYPE.PRE,
    instance: ENV_INSTANCE.CLOUD,
    hosts: ['solomon.cloud-preprod.yandex-team.ru'],
    dcs: [{
      id: 'cloud-preprod',
      title: 'Total',
      host: 'cluster',
      spCluster: 'none',
    }],
    dataOptions: {
      projectId: 'solomon',
      coreCluster: 'preprod',
      alertingCluster: 'preprod',
      hasProcessingShardData: false,
      hasStorageShardData: false,
    },
    infraOptions: YC_SOLOMON_INFRA_OPTIONS.PREPROD,
  },
  CLOUD_PROD: {
    name: 'CLOUD_PROD',
    type: ENV_TYPE.PROD,
    instance: ENV_INSTANCE.CLOUD,
    hosts: [
      'solomon.cloud-prod.yandex-team.ru',
      'solomon.cloud.yandex-team.ru',
    ],
    dcs: [
      {
        id: 'cloud-prod-sas',
        title: 'Sas',
        host: 'Sas',
        spCluster: 'none',
      },
      {
        id: 'cloud-prod-vla',
        title: 'Vla',
        host: 'Vla',
        spCluster: 'none',
      },
    ],
    dataOptions: {
      projectId: 'solomon',
      coreCluster: 'production',
      alertingCluster: 'production',
      hasProcessingShardData: false,
      hasStorageShardData: false,
    },
    infraOptions: YC_SOLOMON_INFRA_OPTIONS.PRODUCTION,
  },
  CLOUD_GPN: {
    name: 'CLOUD_GPN',
    type: ENV_TYPE.PROD,
    instance: ENV_INSTANCE.CLOUD,
    hosts: ['solomon.ycp.gpn.yandexcloud.net'],
    dcs: [{
      id: 'cloud-gpn',
      title: 'Total',
      host: 'cluster',
      spCluster: 'stockpile',
    }],
    dataOptions: {
      projectId: 'bn3g51rigc744ph7skos',
      coreCluster: 'core',
      alertingCluster: 'alerting',
      hasProcessingShardData: false,
      hasStorageShardData: true,
    },
    infraOptions: YC_SOLOMON_INFRA_OPTIONS.GPN,
  },
  CLOUD_IL: {
    name: 'CLOUD_IL',
    type: ENV_TYPE.PROD,
    instance: ENV_INSTANCE.CLOUD,
    hosts: ['solomon.yandexcloud.co.il'],
    dcs: [
      {
        id: 'replica_a',
        title: 'replica_a',
        host: 'cluster',
        spCluster: 'stockpile_a',
      },
      {
        id: 'replica_b',
        title: 'replica_b',
        host: 'cluster',
        spCluster: 'stockpile_b',
      },
    ],
    dataOptions: {
      projectId: 'yc.monitoring.cloud',
      coreCluster: 'production',
      alertingCluster: 'production',
      hasProcessingShardData: false,
      hasStorageShardData: true,
    },
    infraOptions: YC_SOLOMON_INFRA_OPTIONS.ISRAEL,
  },
};

export const ENVIRONMENTS_LIST = values(ENVIRONMENTS);

const DEFAULT_ENV = ENVIRONMENTS.PRODUCTION;

function guessEnv() {
  const { host } = window.location;

  for (let i = 0; i < ENVIRONMENTS_LIST.length; ++i) {
    const env = ENVIRONMENTS_LIST[i];
    if (env.hosts.indexOf(host) >= 0) {
      return env;
    }
  }

  return DEFAULT_ENV;
}

const env = guessEnv();

export function getEnv() {
  return env;
}

export const knownDcs = env.dcs;
export const projectForEnv = env.dataOptions.projectId;
export const coreClusterForEnv = env.dataOptions.coreCluster;
export const alertingClusterForEnv = env.dataOptions.alertingCluster;
export const hasProcessingShardDataForEnv = env.dataOptions.hasProcessingShardData;
export const hasStorageShardDataForEnv = env.dataOptions.hasStorageShardData;

export function isAsUserAuth() {
  return isLocal || env.type === ENV_TYPE.DEV;
}

export function isTestEnv() {
  return env.type === ENV_TYPE.TEST;
}

export function isPreEnv() {
  return env.type === ENV_TYPE.PRE;
}

export function isProdEnv() {
  return env.type === ENV_TYPE.PROD;
}

// For redirect to Monitoring only.
export function isPreOrProdIntEnv() {
  return env === ENVIRONMENTS.PRESTABLE || env === ENVIRONMENTS.PRODUCTION;
}

export function isCloud() {
  return env.instance === ENV_INSTANCE.CLOUD;
}

export function isCloudGpn() {
  return env.name === 'CLOUD_GPN';
}

export function isCloudIL() {
  return env.name === 'CLOUD_IL';
}

export default env;
