/* eslint-disable no-continue,no-cond-assign */
import entries from 'lodash/entries';

const LABEL_PREFIX = 'l.';

const PROHIBITED_LABEL_NAMES = [
  'short_name', 'DC', 'summary', 'deriv', 'project', 'cluster', 'service',
  'description', 'graphMode', 'secondaryGraphMode', 'stack', 'scale',
  'interpolate', 'aggr', 'dropNaN', 'numberFormat', 'cs',
  'green', 'yellow', 'red', 'violet', 'greenValue', 'yellowValue', 'redValue', 'violetValue',
  'norm', 'min', 'max', 'orderBy', 'asc', 'limit', 'transform', 'movingWindow', 'movingPercentile',
  'overLinesTransform', 'percentiles', 'bucketLabel', 'filter', 'filterBy', 'filterLimit',
  'downsampling', 'grid', 'maxPoints', 'downsamplingAggr', 'downsamplingFill', 'ignoreInf', 'ignoreMinStepMillis', 'hideNoData',
  'threshold', 'graphOnly', 'points', 'forceHost', 'forceCluster', 'reviverClientId', 'graph',
  'dashboard', 'trigger', 'triggers', 'legend', 'autorefresh', 'b', 'e',
  'checks', 'backgroudQuery', 'clientMetaDataVersionHash',
  'solomonClientVersion', 'showCensoredIntervals', 'dataDelay', 'countUsage',
  'chartBorderWidth', 'distributionMode', '__experimental', '__mode', '__old', '__force_new', 'utm_source',
];

function isShardKey(key) {
  return key === 'project' || key === 'cluster' || key === 'service';
}

/**
 * Get selectors from url query
 *
 * @param {URLSearchParams} parsedSearch
 * @param keepProject
 * @param keepClusterService
 */
function getSelectorsFromQueryArgs(parsedSearch, keepProject = false, keepClusterService = false) {
  const result = [];

  const it = parsedSearch.entries();

  let entry;

  const knownKeys = new Set();

  while (!((entry = it.next()).done)) {
    const keyValue = entry.value;
    const key = keyValue[0];
    const value = keyValue[1];

    if (!value) {
      continue;
    }

    let fixedKey;

    if (key.startsWith(LABEL_PREFIX)) {
      const keyWithoutPrefix = key.substr(LABEL_PREFIX.length);

      if (isShardKey(keyWithoutPrefix)) {
        continue;
      } else {
        fixedKey = keyWithoutPrefix;
      }
    } else if ((keepProject && key === 'project')
      || (keepClusterService && (key === 'cluster' || key === 'service'))
      || PROHIBITED_LABEL_NAMES.indexOf(key) < 0) {
      fixedKey = key;
    } else {
      continue;
    }

    if (!knownKeys.has(fixedKey)) {
      knownKeys.add(fixedKey);
      result.push([fixedKey, value]);
    }
  }

  return result;
}

/**
 * Get selectors from url query
 *
 * @param params
 * @return {Array} selectors as entries
 */
export function getSelectorsAsEntriesFromParams(params, useProject = false) {
  const result = [];

  const cluster = params.cluster || '';
  const service = params.service || '';
  const project = params.project || '';

  const paramEntries = entries(params);

  const knownKeys = new Set();

  // eslint-disable-next-line no-restricted-syntax
  for (const paramEntry of paramEntries) {
    const key = paramEntry[0];
    const value = paramEntry[1];

    if (!value) {
      continue;
    }

    let fixedKey;

    if (isShardKey(key)) {
      continue;
    }
    if (key.startsWith(LABEL_PREFIX)) {
      fixedKey = key.substr(LABEL_PREFIX.length);
    } else if (PROHIBITED_LABEL_NAMES.indexOf(key) < 0) {
      fixedKey = key;
    } else {
      continue;
    }

    if (!knownKeys.has(fixedKey)) {
      knownKeys.add(fixedKey);
      result.push([fixedKey, value]);
    }
  }

  if (service) {
    result.unshift(['service', service]);
  }
  if (cluster) {
    result.unshift(['cluster', cluster]);
  }
  if (useProject && project) {
    result.unshift(['project', project]);
  }
  return result;
}

export default getSelectorsFromQueryArgs;
