/* eslint-disable import/no-cycle */
import UrlUtils from './UrlUtils';
import QueryArg from './QueryArg';
import parseBoolean, { isTrue } from '../boolean';
import LinkedHashMap from '../LinkedHashMap';

class UrlParsed {
  constructor(beforeQuery, queryArgs) {
    this.beforeQuery = beforeQuery;
    this.queryArgs = queryArgs;
  }

  hasArg(name) {
    for (let i = 0; i < this.queryArgs.length; ++i) {
      const queryArg = this.queryArgs[i];
      if (queryArg.name.getDecoded() === name) {
        return true;
      }
    }
    return false;
  }

  reconstruct() {
    if (this.queryArgs.length === 0) {
      return this.beforeQuery;
    }

    let r = '';
    r += this.beforeQuery;
    r += this.getQuery();
    return r;
  }

  getBeforeQuery() {
    return this.beforeQuery;
  }

  getQuery() {
    return `?${UrlUtils.makeQueryArgsFromNvPairs(this.queryArgs)}`;
  }

  getLogicalQueryArgOrNull(name) {
    for (let i = this.queryArgs.length - 1; i >= 0; --i) {
      const queryArg = this.queryArgs[i];
      if (queryArg.name.getDecoded() === name) {
        return queryArg.value.getDecoded();
      }
    }

    return null;
  }

  getLogicalQueryArgOrEmpty(name) {
    const r = this.getLogicalQueryArgOrNull(name);
    return r !== null && r !== undefined ? r : '';
  }

  getLogicalQueryArgBoolean(name) {
    const value = this.getLogicalQueryArgOrEmpty(name);
    return isTrue(value);
  }

  getLogicalQueryArgNonNegativeIntegerOr(name, defaultValue) {
    const value = this.getLogicalQueryArgOrEmpty(name);

    if (value && !isNaN(value)) {
      return parseInt(value, 10);
    }

    return defaultValue;
  }

  getLogicalQueryArgOptBoolean(name) {
    const value = this.getLogicalQueryArgOrEmpty(name);
    return parseBoolean(value);
  }

  getLogicalQueryArgsMapByNames(names) {
    const result = new LinkedHashMap();

    names.forEach((name) => {
      const value = this.getLogicalQueryArgOrEmpty(name);

      if (value) {
        result.put(name, value);
      }
    });

    return result;
  }

  removeQueryArg(name) {
    const r = this.queryArgs.filter((queryArg) => {
      const queryArgDecodedName = queryArg.name.getDecoded();
      return queryArgDecodedName !== name;
    });
    return new UrlParsed(this.beforeQuery, r);
  }

  removeQueryArgs(args) {
    let r = this;
    for (let i = 0; i < args.length; ++i) {
      const arg = args[i];
      r = r.removeQueryArg(arg);
    }

    return r;
  }

  addQueryArg(name, value) {
    const copy = [...this.queryArgs];
    copy.push(QueryArg.decoded(name, value));
    return new UrlParsed(this.beforeQuery, copy);
  }

  updateQueryArg(name, value) {
    const update = QueryArg.decoded(name, value);
    if (!update.name) {
      throw new Error('name is empty');
    }

    const copy = [];
    let updated = false;
    for (let i = 0; i < this.queryArgs.length; ++i) {
      const queryArg = this.queryArgs[i];
      if (queryArg.name.getDecoded() !== update.name.getDecoded()) {
        copy.push(queryArg);
      } else if (!updated) {
        copy.push(update);
        updated = true;
      }
    }

    if (!updated) {
      copy.push(update);
    }

    return new UrlParsed(this.beforeQuery, copy);
  }

  fillDefault(name, value) {
    if (!value) {
      return this;
    }

    if (this.hasArg(name)) {
      return this;
    }

    return this.addQueryArg(name, value);
  }

  fillDefaults(defaults) {
    let r = this;
    for (let i = 0; i < defaults.length; ++i) {
      const nameValue = defaults[i];
      r = r.fillDefault(nameValue.name.getDecoded(), nameValue.value.getDecoded());
    }
    return r;
  }
}

export default UrlParsed;
