/* eslint-disable no-param-reassign,import/no-cycle */
import UrlParsed from './UrlParsed';
import LinkedHashMap from '../LinkedHashMap';
import QueryArg from './QueryArg';
import UserLinksBasic from '../UserLinksBasic';

class UrlUtils {
  static encodeArg(value) {
    return encodeURIComponent(value);
  }

  static decodeArg(value) {
    return decodeURIComponent(value.replace(/\+/g, '%20'));
  }

  static parseUrlQueryArgs(url) {
    if (url == null) {
      return new LinkedHashMap();
    }

    let fixedUrl = url;

    const hashIndex = fixedUrl.indexOf('#');
    if (hashIndex >= 0) {
      fixedUrl = fixedUrl.substring(0, hashIndex);
    }

    const searchIndex = fixedUrl.indexOf('?');
    if (searchIndex < 0) {
      return new LinkedHashMap();
    }
    fixedUrl = fixedUrl.substring(searchIndex + 1);

    return UrlUtils.parseQueryArgs(fixedUrl);
  }

  static parseQueryArgs(args) {
    const result = new LinkedHashMap();

    if (!args) {
      return result;
    }

    const pairs = args.split('&');
    for (let i = 0; i < pairs.length; ++i) {
      const pair = pairs[i];
      const index = pair.indexOf('=');

      if (index >= 0) {
        const key = pair.substring(0, index);
        const value = pair.substring(index + 1);
        result.put(UrlUtils.decodeArg(key), UrlUtils.decodeArg(value));
      }
    }

    return result;
  }

  static updateOrRemoveParameter(url, name, value) {
    if (!value) {
      return UrlUtils.removeQueryArgs(url, name);
    }

    return UrlUtils.updateParameter(url, name, value);
  }

  static updateOrRemoveParameterWithPrefix(url, name, value) {
    if (!value) {
      return UrlUtils.removeQueryArgs(url, name);
    }

    return UrlUtils.updateParameterWithPrefix(url, name, value);
  }

  static updateParameter(url, name, value) {
    return UrlUtils.parseUrl(url)
      .updateQueryArg(name, value)
      .reconstruct();
  }

  static updateParameterWithPrefix(url, name, value) {
    const nameWithPrefix = UserLinksBasic.labelNameToQueryArgName(name);
    const parsedUrl = UrlUtils.parseUrl(url);

    if (parsedUrl.hasArg(nameWithPrefix)) {
      return parsedUrl.updateQueryArg(nameWithPrefix, value).reconstruct();
    }

    return parsedUrl.updateQueryArg(name, value).reconstruct();
  }

  static updateParameters(url, update) {
    let urlParsed = UrlUtils.parseUrl(url);
    const nameValues = update.entries();
    for (let i = 0; i < nameValues.length; ++i) {
      const nameValue = nameValues[i];
      const key = nameValue[0];
      const value = nameValue[1];
      if (value) {
        urlParsed = urlParsed.updateQueryArg(key, value);
      } else {
        urlParsed = urlParsed.removeQueryArg(key);
      }
    }
    return urlParsed.reconstruct();
  }

  static addParameter(url, name, value) {
    if (!url) {
      throw new Error('empty URL');
    }

    let fragment;

    const hash = url.indexOf('#');
    if (hash >= 0) {
      fragment = url.substring(hash);
      url = url.substring(0, hash);
    } else {
      fragment = '';
    }

    const tail = QueryArg.decoded(name, value).reconstruct();

    if (url.endsWith('?') || url.endsWith('&')) {
      return url + tail + fragment;
    }

    if (url.indexOf('?') >= 0) {
      return `${url}&${tail}${fragment}`;
    }

    return `${url}?${tail}${fragment}`;
  }

  static addTwoParameters(url, name1, value1, name2, value2) {
    return UrlUtils.addParameter(UrlUtils.addParameter(url, name1, value1), name2, value2);
  }

  static addThreeParameters(url, name1, value1, name2, value2, name3, value3) {
    let newUrl = UrlUtils.addParameter(url, name1, value1);
    newUrl = UrlUtils.addParameter(newUrl, name2, value2);
    return UrlUtils.addParameter(newUrl, name3, value3);
  }

  static addParameters(url, params) {
    const entries = params.entries();
    for (let i = 0; i < entries.length; ++i) {
      const e = entries[i];
      url = UrlUtils.addParameter(url, e[0], e[1]);
    }
    return url;
  }

  static mapToNvPairs(map) {
    return map.entries()
      .map((entry) => QueryArg.decoded(entry[0], entry[1]));
  }

  static makeQueryArgs(map) {
    return UrlUtils.makeQueryArgsFromNvPairs(UrlUtils.mapToNvPairs(map));
  }

  static makeQueryArgsFromNvPairs(queryArgs) {
    let result = '';

    for (let i = 0; i < queryArgs.length; ++i) {
      if (i !== 0) {
        result += '&';
      }
      const queryArg = queryArgs[i];
      result += queryArg.reconstruct();
    }

    return result;
  }

  static fillDefaultsMap(url, defaults) {
    return UrlUtils.parseUrl(url)
      .fillDefaults(UrlUtils.mapToNvPairs(defaults))
      .reconstruct();
  }

  static parseQueryArgsImpl(queryArgs) {
    const r = [];

    const parts = queryArgs.split('&');
    for (let i = 0; i < parts.length; ++i) {
      const part = parts[i];

      if (part) {
        const eq = part.indexOf('=');
        if (eq < 0) {
          r.push(QueryArg.encoded(part, ''));
        } else {
          const name = part.substring(0, eq);
          const value = part.substring(eq + 1);
          r.push(QueryArg.encoded(name, value));
        }
      }
    }
    return r;
  }

  /**
   *
   * @param url
   * @return {UrlParsed}
   */
  static parseUrl(url) {
    const h = url.indexOf('#');

    if (h >= 0) {
      url = url.substring(0, h);
    }

    const q = url.indexOf('?');
    if (q < 0) {
      return new UrlParsed(url, []);
    }

    const query = url.substring(q + 1);
    const queryArgs = UrlUtils.parseQueryArgsImpl(query);

    return new UrlParsed(url.substring(0, q), queryArgs);
  }

  static removeQueryArgs(url, ...names) {
    const q = url.indexOf('?');
    if (q < 0) {
      return url;
    }

    return UrlUtils.parseUrl(url)
      .removeQueryArgs(names)
      .reconstruct();
  }
}

export default UrlUtils;
