/* eslint-disable no-plusplus */
import flatMap from 'lodash/flatMap';
import uniqWith from 'lodash/uniqWith';

function isLetter(c) {
  return c.toLowerCase() !== c.toUpperCase();
}

function isDigit(c) {
  return c >= '0' && c <= '9';
}

class WordsFromLabelValueParser {
  static parse(input) {
    return new WordsFromLabelValueParser(input).parseImpl();
  }

  static parsePrefixes(input) {
    const parseResult = this.parse(input);

    const prefixes = flatMap(parseResult.map((wap) => [wap[1], `${wap[1]}${wap[0]}`]));

    return uniqWith(prefixes);
  }

  constructor(labelValue) {
    this.data = labelValue;
    this.buffer = new Array(this.data.length);
    this.bufferIdx = 0;
    this.generatedEmptyPrefix = false;
  }

  parseImpl() {
    const collector = [];

    let prefixLength = 0;

    for (let i = 0; i < this.data.length; ++i) {
      const c = this.data[i];
      if (isLetter(c) || isDigit(c)) {
        this.tryAppend(c, collector, prefixLength);
      } else {
        this.flushBuffer(collector, prefixLength);
      }
      prefixLength++;
    }
    this.flushBuffer(collector, prefixLength);
    if (!this.generatedEmptyPrefix) {
      collector.push(['', '']);
    }
    return collector;
  }

  flushBuffer(collector, prefixLength) {
    if (prefixLength === this.bufferIdx) {
      this.generatedEmptyPrefix = true;
    }
    if (this.bufferIdx > 0) {
      collector.push([
        this.buffer.slice(0, this.bufferIdx).join(''),
        this.data.substring(0, prefixLength - this.bufferIdx),
      ]);

      this.bufferIdx = 0;
    }
  }

  tryAppend(c, collector, prefixLength) {
    if (this.bufferIdx === 0) {
      this.append(c);
    } else {
      const lastChar = this.buffer[this.bufferIdx - 1];
      if (isDigit(lastChar)) {
        if (isDigit(c)) {
          this.append(c);
        } else {
          this.flushBuffer(collector, prefixLength);
          this.tryAppend(c, collector, prefixLength);
        }
      } else if (isLetter(lastChar) && lastChar === lastChar.toLowerCase()) {
        if (isLetter(c) && c === c.toUpperCase()) {
          this.flushBuffer(collector, prefixLength);
          this.tryAppend(c, collector, prefixLength);
        } else {
          this.append(c);
        }
      } else {
        this.append(c);
      }
    }
  }

  append(c) {
    this.buffer[this.bufferIdx++] = c;
  }
}

export default WordsFromLabelValueParser;
