# -*- coding: utf-8 -*-
"""
График для телевизоров SAMSUNG.
"""
import sys
import getopt
from plotter.share.graph.viewport import iviewport
from plotter.share.cairo import image
from plotter.share.cairo.canvas import Canvas
from plotter.share.plot.window import discrete_x_lh, window_from_lh, calc_y_lh
from plotter.share.plot.grid import draw_grid


def plot_from_file(f, func=float):
    """
    Читает график из файла.
    """
    return [[func(c) for c in s.strip().split(' ')] for s in f]


# Ширина выходной картинки и отступы.
W = 560
H = 340
BORDER = 20
PADDING_TOP = 5
PADDING_BOTTOM = 10
PADDING_LEFT = 0

# Цвета: цвет сетки, цвет графика, цвет заливки области под графиком.
GRID_COLOR = (0.89, 0.85, 0.85, 1.0)
TEXT_COLOR = (0.447, 0.447, 0.447, 1.0)
GRAPH_COLOR = (1.0, 0, 0, 1.0)
FILL_COLOR = (1.0, 0, 0, 0.1)

# Ширина линий.
GRID_LINE_WIDTH = 1.0
GRAPH_LINE_WIDTH = 2.8

CICLE_RADIUS = 4
CICLE_LINE_WIDTH = 2.0

# Подписи.
FONT = "shadow_0.ttf"
FONT_SIZE = 12
# Смещение надписей по оси X
TEXT_Y_PADDING = 14

# Если True, то рисуется фрейм (для отладки)
DRAW_FRAME = False

# По умолчанию рисуем вот столько точек.
N = 10


def draw_samsung_plot(xlabels, plot, font, fontsize):
    """
    Рисует график для телика LG по точкам PLOT.
    """

    # Создаем порт просмотра и окно
    v = iviewport((PADDING_LEFT + BORDER, PADDING_TOP), (W - BORDER, H - BORDER - PADDING_BOTTOM))
    x_lh = discrete_x_lh(plot)
    y_lh = calc_y_lh(plot)
    w = window_from_lh(x_lh, y_lh)

    surface, ctx = image.image(W, H)
    try:
        if DRAW_FRAME:
            ctx.rectangle(0, 0, W, H)

        canvas = Canvas(ctx, w, v)
        canvas.set_font_from_file(font, fontsize)

        # Рисуем сетку.
        ctx.set_line_width(GRID_LINE_WIDTH)
        draw_grid(xlabels, canvas, x_lh, y_lh, GRID_COLOR, TEXT_COLOR, TEXT_Y_PADDING)
        ctx.stroke()

        # r - это радиус маркера.
        r = CICLE_RADIUS
        crop = (r, r)

        # Рисуем заливку под графиком.
        ctx.set_source_rgba(*FILL_COLOR)
        prev = (0, y_lh[1])
        canvas.move_to(prev)
        for x, y in plot:
            canvas.line_to((x, y))
            canvas.arcPi((x, y), r)
            canvas.line_to((x, y))
        canvas.line_to((x_lh[2], y_lh[1]))
        ctx.close_path()
        ctx.fill()
        ctx.stroke()

        # Рисуем график.    
        ctx.set_source_rgba(*GRAPH_COLOR)
        prev = None
        for x, y in plot:
            ctx.set_line_width(CICLE_LINE_WIDTH)
            canvas.cicle((x, y), r)
            ctx.stroke()
            ctx.set_line_width(GRAPH_LINE_WIDTH)
            if prev is not None:
                canvas.line(prev, (x, y), crop)
                ctx.stroke()
            prev = (x, y)
    finally:
        del ctx

    return surface


def usage():
    print("usage: %s [-f font] [-s fontsize] [-n points-count] [-i input-file] [-o output-file]" % sys.argv[0])


if __name__ == "__main__":
    try:
        opts, args = getopt.getopt(sys.argv[1:], "f:s:n:i:o:", [])
    except getopt.GetoptError as err:
        # print help information and exit:
        print(str(err))  # will print something like "option -a not recognized"
        usage()
        sys.exit(2)

    n = N
    input_file = sys.stdin
    close_input = False
    output_file = None
    font = FONT
    fontsize = FONT_SIZE

    for opt, val in opts:
        if opt in ["-n"]:
            n = int(val)
        elif opt in ["-i"]:
            input_file = open(val, "r")
            close_input = True
        elif opt in ["-o"]:
            output_file = val
        elif opt in ["-f"]:
            font = val
        elif opt in ["-s"]:
            fontsize = int(val)

    if output_file is None:
        print("Output file is None.")
        exit(2)

    # Перенумеровываем точки нафиг.
    try:
        plot_b = plot_from_file(sys.stdin)[-n:]
        plot = []
        xlabels = []
        i = 0
        for ut, y in plot_b:
            plot.append((i, y))
            xlabels.append(ut)
            i += 1

        if len(plot) == 0:
            exit(0)

        surface = draw_samsung_plot(xlabels, plot, font, fontsize)
        surface.write_to_png(output_file)
    finally:
        if close_input:
            input_file.close()
