# -*- coding: utf-8 -*-
"""
Рисование с использованием viewport'а.
Здесь точка представляется кортежем из двух элементов: (x,y).
"""
from math import pi
from plotter.share.graph.viewport import WVMapper, X, Y, MIN, MAX
from plotter.share.cairo.font import create_cairo_font_face_for_file
from plotter.share.math.epsilon import epsilon
from plotter.share.graph.crop import crop_cicle

__all__ = ['Canvas']

__author__ = "Zasimov Alexey"
__email__ = "zasimov-a@yandex-team.ru"


def cairo_text_props(ctx, text):
    """
    Вычисляем размер текста для контекста ctx.

    Возвращает x_bearing, y_bearing, width, height.
    """
    return ctx.text_extents(text)[:4]


class Canvas(object):
    """
    Рисуемся =)
    """

    def __init__(self, ctx, window, viewport):
        self.ctx = ctx
        self.mapper = WVMapper(window, viewport)

    def xranges(self):
        return self.mapper.w[MIN][X], self.mapper.w[MAX][X]

    def yranges(self):
        return self.mapper.w[MIN][Y], self.mapper.w[MAX][Y]

    def move_to(self, s):
        ss = self.mapper(*s)
        self.ctx.move_to(*ss)

    def line(self, s, e, crop=None):
        ss = self.mapper(*s)
        es = self.mapper(*e)

        # Реализуем отсечение.
        if crop is not None:
            ss, es = crop_cicle(ss, es, crop)

        self.ctx.move_to(*ss)
        self.ctx.line_to(*es)

    def line_to(self, p):
        p = self.mapper(*p)
        self.ctx.line_to(*p)

    def arcPi(self, c, r):
        c = self.mapper(*c)
        self.ctx.arc(c[X], c[Y], r, 0, pi)

    def cicle(self, c, r, stroke=True):
        c = self.mapper(*c)
        self.ctx.move_to(*c)
        if stroke:
            self.ctx.stroke()
        self.ctx.arc(c[X], c[Y], r, 0, 2*pi)

    def show_text(self, p, text, padding_x=0, padding_y=0):
        pp = self.mapper(*p)
        self.ctx.move_to(pp[X]+padding_x, pp[Y]+padding_y)
        self.ctx.show_text(text)

    def set_font_from_file(self, filename, font_size):
        face = create_cairo_font_face_for_file(filename, 0)
        self.ctx.set_font_face(face)
        self.ctx.set_font_size(font_size)
        return face

    def text_x_center(self, p, text, padding_y=0.0):
        """
        Настраивает контекст так, чтобы текст рисовался по центру относительно
        точки p. Выравнивание идет только по координате x.
        """
        p = self.mapper(*p)
        x_bearing, y_bearing, width, height = cairo_text_props(self.ctx, text)
        self.ctx.move_to(p[0] - width / 2 + x_bearing, p[1] + padding_y)
        self.ctx.show_text(text)
