# -*- coding: utf-8 -*-

__author__ = "Zasimov Alexey"
__email__ = "zasimov-a@yandex-team.ru"


import time
from os import environ

try:
    from contextlib import contextmanager
except ImportError:
    contextmanager = None


_tzset = time.tzset


def tzset(tz):
    """
    Устанавливает временную зону tz.

    @param tz: Человеческое имя зона (например, Europe/Moscow).
    """
    if tz is None:
        if "TZ" in environ:
            del environ["TZ"]
    else:
        environ["TZ"] = tz
    return _tzset()


def tzpush(tz):
    previous = environ.get('TZ', None)
    tzset(tz)
    return previous

def tzpop(previous):
    tzset(previous)


def withtz(tz, func, *args, **kwargs): 
    """
    Запускает функцию func в окружении с установленной временной зоной tz.

    >>> with_tz("Asia/Nicosia", lambda x: time.timezone, None)
    -7200

    @param tz: Человеческое имя зоны (например, Europe/Moscow).
    @param func: Функция для запуска.
    @param args: Аргументы для функции func.
    """
    previous = tzpush(tz)
    try:
        return func(*args, **kwargs)
    finally:                                                                                                                
        tzpop(previous)

if contextmanager is not None:
    @contextmanager
    def timezone(tz):
        previous = tzpush(tz)
        try:
            yield
        finally:
            tzpop(previous)
else:
    timezone = None
