# -*- coding: utf-8 -*-

"""
Калькулятор, вычисляющий значение unixtime для котировки.
"""
from __future__ import print_function
from stocks3.share.dateandtime import to_unixtime
from stocks3.core.calculator import Calculator
from stocks3.core.factories import calculators
from stocks3.share.timezone import get_timezone
from stocks3.core.stock import Quote
from stocks3.core.sharedregion import SHARED_REGION


__author__ = "Zasimov Alexey"
__email__ = "zasimov-a@yandex-team.ru"


class UnixtimeCalculator(Calculator):
    """
    Добавляет к объекту поле unixtime, содержащее unixtime :)
    """
    def makeConfig(self):
        Calculator.makeConfig(self)
        # Читаем список котировок, для которых необходимо установить 0
        # Это делается для совместимости со старыми экспортирующими скриптами
        self.zero_quotes = []
        for node in self.node.findall("set-zero/quote"):
            quoteName = node.attrib["name"]
            quote = self.default.quotes_config.getQuote(quoteName)
            self.zero_quotes.append(quote.quote_id)

    def get_timezone(self, source):
        tz = get_timezone(self, source)
        # TODO: сделать проверку корректности временной зоны
        return tz

    def calc(self, source, price):
        Calculator.calc(self, source, price)
        if price.quote.quote_id in self.zero_quotes:
            unixtime = 0
        else:
            tz = self.get_timezone(source)
            tz = price.tz if price.tz is not None else tz
            unixtime = to_unixtime(price.date, tz)
        price.unixtime = int(unixtime)
        return price


class DropUnixtimeCalculator(Calculator):
    """
    Устанавливает флажок drop_unixtime, которой говорит шаблонизатору о том,
    что в экспорте unixtime нужно сбросить.
    """
    def calc(self, source, price):
        Calculator.calc(self, source, price)
        assert isinstance(price.quote, Quote), "DropUnixtimeCalculator - only for export!."
        quote_info = source.parser.getQuoteInfo(price.getRegion(SHARED_REGION), price.quote)
        drop_unixtime = source.parser.drop_unixtime or quote_info.get("drop_unixtime", False)
        self.save_value(price, "drop_unixtime", drop_unixtime)
        return price


calculators.register("stocks3.calculators.Unixtime", UnixtimeCalculator)
calculators.register("stocks3.calculators.DropUnixtime", DropUnixtimeCalculator)


