# -*- coding: utf-8 -*-

"""
Модуль проверяет значения по календарю - позволяет проходить котировкам только в рабочие дни (по россии)
"""
from __future__ import print_function
from datetime import datetime
from stocks3.core.checker import Checker
from stocks3.core.factories import checkers

__author__ = "Bogdanov Evgeny"
__email__ = "evbogdanov@yandex-team.ru"


class TimeFilterChecker(Checker):
    """
    Проверяет новое значение на пригодность.

    """
    mignight_time = datetime.strptime("00:00:00", "%H:%M:%S").time()

    def __init__(self, tree, node, source):
        Checker.__init__(self, tree, node, source)

    def makeConfig(self):
        self._load_ids_checks()
        return self

    def _load_ids_checks(self):
        """
        Загружает ограничения для конкретных котировок.
        Обрабатывает элементы checkers/checker.
        """
        self._ids_checks = {}
        for checkNode in self.node.findall("check"):
            self._ids_checks[checkNode.attrib["quote"]] = {
                'from': datetime.strptime(checkNode.attrib["time_from"], "%H:%M:%S").time(),
                'to': datetime.strptime(checkNode.attrib["time_to"], "%H:%M:%S").time()
            }

    @staticmethod
    def perform_check(date, time_from, time_to):
        quote_time = date.time()
        if time_to == TimeFilterChecker.mignight_time:
            return time_from <= quote_time or TimeFilterChecker.mignight_time == quote_time
        return (time_from <= quote_time) and (quote_time <= time_to)

    def check(self, price):
        """
        Получает предыдущее значение котировки и, если оно есть, выполняет
        проверку.
        """
        quotename = self.source.default.quotes_config.getQuoteName(price.quote.quote_id)
        if quotename in self._ids_checks:
            return self.perform_check(price.date, self._ids_checks[quotename]['from'],
                                      self._ids_checks[quotename]['to'])
        return True


checkers.register("stocks3.checkers.timeFilter", TimeFilterChecker)
