# -*- coding: utf-8 -*-

"""
Конфигурация модулей Finam.
"""

__author__ = "Zasimov Alexey"
__email__ = "zasimov-a@yandex-team.ru"


from stocks3.core.config import Configurable, parse_xml
from datetime import datetime


class FinamConfig(Configurable):
    """
    Дополнительная конфигурация для модулей Finam. Этот класс умеет вычислять
    URL для получения данных.
    """

    def __init__(self, filename):
        tree = parse_xml(filename)
        root = tree.getroot()
        Configurable.__init__(self, tree, root)

    def makeConfig(self):
        self.url_template = self.readString("", "url-template").strip()
        self.period = self.readString("", "period").strip()
        self.markets = {}  # значения для параметра market
        self._read_markets()

    def _read_markets(self):
        for market in self.node.findall("markets/market"):
            self.markets[market.attrib["type"]] = market.text.strip()

    def _make_date_variables(self, prefix, dt):
        """
        В шаблон URL вставляются даты полностью, а также по отдельности день,
        месяц и год. Интересная система :) Но ничего не поделаешь.
        """
        assert type(dt) == datetime
        variables = {"%s_date" % prefix: dt.strftime("%y%m%d"), "%s_day" % prefix: str(dt.day),
                     "%s_month" % prefix: str(dt.month - 1), "%s_year" % prefix: str(dt.year)}
        return variables

    def prepareURL(self, market, short_name, quote_id, from_date, to_date):
        """
        Конструирует адрес для получения данных о котировках.
        """
        # Создаем словарик, значения которого подставятся в url-template
        variables = {"short_name": short_name,
                     "id": str(quote_id),
                     "period": self.period,
                     "market": self.markets[market]}
        variables.update(self._make_date_variables('f', from_date))
        variables.update(self._make_date_variables('t', to_date))
        url = self.url_template
        for var, value in variables.items():
            url = url.replace('{'+var+'}', value)
        return url

