# -*- coding: utf-8 -*-
"""
Разбираем ставку рефинансирования.
"""
from __future__ import print_function
import datetime
import html.parser
import re
from stocks3.core.parser import Parser
from stocks3.core.place import FilePlace
from stocks3.core.factories import parsers

__author__ = "Zasimov Alexey"
__email__ = "zasimov-a@yandex-team.ru"


class MyHTMLParser(html.parser.HTMLParser):
    def __init__(self):
        html.parser.HTMLParser.__init__(self, convert_charrefs=True)
        self._data = [] # return object array
        self._curr_data = {} # current table line parsed data

        self._in_cbr_data_table = False
        self._in_cbr_table_n = 0
        self._cbr_data_table_lno = 0  # line number counter
        self._cbr_data_table_cno = 0  # column number counter
        self._in_td_elem = False  # if we are in cbr data table td element

        self._cno2name = {1: 'dates', 2: 'value', 3: 'law'}

        self._name2month = {'января': 1, 'февраля': 2, 'марта': 3, 'апреля': 4, 'мая': 5, 'июня': 6, 'июля': 7,
                            'августа': 8, 'сентября': 9, 'октября': 10, 'ноября': 11, 'декабря': 12}

    def handle_starttag(self, tag, attrs):
        if tag == 'table':
            self._in_cbr_table_n += 1
            if 1 == self._in_cbr_table_n:
                self._in_cbr_data_table = True

        if self._in_cbr_data_table:
            if tag == 'tr':  # increase line number counter
                self._cbr_data_table_lno += 1
                self._cbr_data_table_cno = 0
                self._curr_data = {'dates': '', 'value': '', 'law': ''}

            if tag == 'td':
                self._cbr_data_table_cno += 1
                self._in_td_elem = True

    def handle_endtag(self, tag):
        if self._in_cbr_data_table:

            if tag == 'table':
                self._in_cbr_data_table = False

            if tag == 'td':
                self._in_td_elem = False

                if self._cbr_data_table_lno > 1:
                    col_name = self._cno2name[self._cbr_data_table_cno]
                    data = self._curr_data[col_name]
                    if col_name == 'dates':
                        self._curr_data[col_name] = self.parseDate(data)

                    if col_name == 'value':
                        self._curr_data[col_name] = data.strip().replace(',', '.')

                    if col_name == 'law':
                        self._curr_data[col_name] = data.strip()

            if tag == 'tr':
                if self._cbr_data_table_lno == 1:
                    return

                for d in self._curr_data['dates']:
                    if (d):
                        item = {'time': '00:00', 'date': d.strftime('%d.%m.%Y'), 'value': self._curr_data['value']}
                        self._data.append(item)

    def handle_data(self, data):
        if self._in_cbr_data_table:
            if self._cbr_data_table_lno == 1:
                return
            else:
                if self._in_td_elem:
                    col_name = self._cno2name[self._cbr_data_table_cno]

                    self._curr_data[col_name] += data

    # parse date string like: 30 апреля 2010 г. – 31 мая 2010 г.
    def parseDate(self, data):
        # buf = data.decode('cp1251')
        buf = data
        buf = buf.strip()

        parts = buf.split(u'–')

        date1 = self.parseDateString(parts[0])
        date2 = None

        if len(parts) > 1 and len(parts[1]) > 0:
            date2 = self.parseDateString(parts[1])

        if not date2:
            return [date1]

        res = [date1, date2]
        return res

    def parseDateString(self, buf):
        buf = buf.strip()
        m = re.search(u'^(\d+)[\s]*(\S+)[\s]+(\d\d\d\d)[\s]*г.', buf, re.UNICODE)

        if m:
            day = int(m.group(1))
            year = int(m.group(3))
            month_name = m.group(2)
            month = self._name2month[month_name]
            res = datetime.date(year, month, day)

            s = res.strftime("%d.%m.%Y")
            return res

    def parse(self, input_file):
        f = open(input_file, 'r', encoding="cp1251")
        file_data = f.read()
        f.close()
        self.feed(file_data)

    def getData(self):
        return self._data

    def error(self, message):
        pass


class CbrRefinancingParser(Parser):
    places = [FilePlace]

    def parse(self, place):
        oldParser = MyHTMLParser()
        oldParser.parse(place.filename)
        for data in oldParser.getData():
            sourceDate = data['date']
            value = float(data['value'])
            price = self.default_map.make_price_date('all', sourceDate, value)
            if price is not None:
                yield price


parsers.register("stocks3.parsers.CbrRefinancing", CbrRefinancingParser)
