# -*- coding: utf-8 -*-

"""
Парсим рейтинг банков от ds.ifx.banks.rating.
"""

__author__ = "Zasimov Alexey"
__email__ = "zasimov-a@yandex-team.ru"


from stocks3.parsers.xml import XMLParser
from stocks3.core.factories import parsers


# possible status values: установлен, повышен, понижен, подтвержден, снят
STATUS2NUM  = { u'установлен' : 1,
    u'повышен' : 2,
    u'понижен' : 3,
    u'подтвержден' : 4,
    u'снят' : 5 }

# dict: rating string to num value
RATING2NUM = { "Aaa.ru" : 1,
    "Aa1.ru" : 2,
    "Aa2.ru" : 3,
    "Aa3.ru" : 4,
    "A1.ru" : 5,
    "A2.ru" : 6,
    "A3.ru" : 7,
    "Baa1.ru" : 8,
    "Baa2.ru" : 9,
    "Baa3.ru" : 10,
    "Ba1.ru" : 11,
    "Ba2.ru" : 12,
    "Ba3.ru" : 13,
    "B1.ru" : 14,
    "B2.ru" : 15,
    "B3.ru" : 16,
    "Caa1.ru" : 17,
    "Caa2.ru" : 18,
    "Caa3.ru" : 19,
    "Ca.ru" : 20,
    "C.ru" : 21
}


class DsIfxBanksRatingParser(XMLParser):
    """
    Парсит рейтинг банков.
    """

    def _calc_rating_value(self, change, rating):
        """
        Вычисляем рейтинг банка (по странной формуле).

        Возвращает None, если по каким-то причинам вычислить рейтинг не
        удалось. Этот банк пропускается.
        """
        status = change.lower()

        if status not in STATUS2NUM:
            return None

        if rating not in RATING2NUM:
            return None

        status_value = STATUS2NUM[status]
        rating_value = RATING2NUM[rating]

        res = (0 | rating_value) | ((status_value << 24) & 0xFF000000)

        return float(res)


    def parse(self, place):
        XMLParser.parse(self, place)
        for item in self.dataRoot.findall("ITEM"):
            sourceQuoteId = item.attrib["id"]
            sourceDate = item.attrib["date"]
            value = self._calc_rating_value(item.attrib["change"], item.attrib["rating"])
            if value is None:
                continue
            price = self.default_map.make_price_date(sourceQuoteId, sourceDate, value)
            if price is not None:
                yield price


parsers.register("stocks3.parsers.DsIfxBanksRating", DsIfxBanksRatingParser)


