# -*- coding: utf-8 -*-

"""
Разбор данных от finance.ua.
"""

__author__ = "Zasimov Alexey"
__email__ = "zasimov-a@yandex-team.ru"


from stocks3.transports.finance_ua import SpecialFinanceUaPlace
from stocks3.core.parser import Parser
from stocks3.core.factories import parsers


class FinanceUaParser(Parser):
    """
    Этот парсер и не парсер ни какой. Транспорт отдает в специально
    подготовленное место (L{SpecialFinanceUaPlace}) уже отпарсенные данные. Нам
    нужно их завернуть в L{Price}.
    """

    places = [SpecialFinanceUaPlace]

    def makeConfig(self):
        Parser.makeConfig(self)
        self._read_regions_map()

    def _read_regions_map(self):
        # Загружаем карту регионов
        self.regions_map = {}
        for region_node in self.node.findall("regions-map/region"):
            from_ = int(region_node.attrib["from"])
            to = int(region_node.attrib["to"])
            self.regions_map[from_] = to

    def _process_data(self, source_data, use_regions_map):
        """
        Обрабатываем данные. Так как из источника приходят данные для регионов
        и Украины в аналогичных форматах, используем для их разбора эту
        функцию. Разница лишь в том, что при разборе регионов котировки
        привязываются к конкретному региону (use_regions_map = True в этом
        случае).
        """
        sourceDate = source_data['date']
        for data in source_data['currency']:
            sourceQuoteId = data['alphabetic_code']
            buy_value = float(data['ask'])
            sell_value = float(data['bid'])

            # Смотрим, в какой регион попадет наше значение
            if use_regions_map:
                raw_region = data.get('region', None)
                if raw_region is None:
                    # Пропускаем котировки без регионов
                    continue
                region = self.regions_map[int(raw_region)]
            else:
                region = None

            price = self.default_map.make_dual_price(sourceQuoteId, sourceDate, buy_value, sell_value)

            if price is not None:
                if region is not None:
                    price.onlyForRegion(region)
                yield price

    def parse(self, place):
        Parser.parse(self, place)
        for price in self._process_data(place.regions_data, True):
            yield price
        for price in self._process_data(place.ukraine_data, False):
            yield price


parsers.register("stocks3.parsers.FinanceUa", FinanceUaParser)

