# -*- coding: utf-8 -*-
"""
Просто функция для получения временной зоны источника. Алгоритм работы прост:
    1. Пытаемся прочитать зону из ветки текущего объекта (параметр timezone).
    2. Пытаемся прочитать зону из самого вернего уровня дерева конфигурации.
    3. Если шаг 1 и шаг 2 не дали результата - берем зону по умолчанию
    (L{Default}).
"""
from stocks3.core.source import BaseSource, JsonSource, Source
from stocks3.core.stock import Quote
from stocks3.core.config import Configurable, read_attrib_or_text
from stocks3.core.default import Default

__author__ = "Zasimov Alexey"
__email__ = "zasimov-a@yandex-team.ru"


def get_timezone(configurable, source):
    if configurable is not None:
        assert isinstance(configurable, (Configurable, Quote)), "Expected configurable object or None for get_timezone."
    assert isinstance(source, BaseSource), "Expected source object for get_timezone."
    # Пытаемся получить временную зону, установленную для текущего объекта
    tz = None
    if configurable is not None:
        tz = configurable.readString("", "timezone", "") or None
    if tz is None:
        # Такой зоны нет - пробуем получить временную зону из дерева источника
        if isinstance(source, JsonSource):
            tz = source.timezone or None
        else:
            assert isinstance(source, Source), "Expected source object for get_timezone."
            tz = read_attrib_or_text(source.node, "timezone", "") or None
        if tz is None:
            default = source.default
            # И такой зоны нет - берем временную зону по умолчанию
            tz = default.tz
    return tz


def get_timezone_fixed(quote, source):
    if quote is not None:
        assert isinstance(quote, Quote), "Expected Quote object or None for get_timezone."
    assert isinstance(source, BaseSource), "Expected Source object for get_timezone."
    # Пытаемся получить временную зону, установленную для котировки в конфиге котировок
    tz = None
    if quote is not None:
        tz = quote.timezone

    if isinstance(source, JsonSource):
        for map_name, idmap in source.parser.maps.items():
            for quote_info in idmap.values():
                map_quote = quote_info["quote"]
                if map_quote is not None and map_quote.quote_id == quote.quote_id:
                    if quote_info["tz"] is not None:
                        # Нашли котировку в карте и для нее установлена особенная временная зона.
                        tz = quote_info["tz"]
                        break
        if tz is None:
            # Такой зоны нет - пробуем получить временную зону из дерева источника
            tz = source.timezone or None
    else:
        assert isinstance(source, Source), "Expected Source object for get_timezone."
        for map_name, map_ in source.parser.maps.items():
            for from_, quote_info in map_.items():
                map_quote = quote_info["quote"]
                if map_quote is not None and map_quote.quote_id == quote.quote_id:
                    if quote_info["tz"] is not None:
                        # Нашли котировку в карте и для нее установлена особенная временная зона.
                        tz = quote_info["tz"]
                        break

        if tz is None:
            # Такой зоны нет - пробуем получить временную зону из дерева источника
            tz = read_attrib_or_text(source.node, "timezone", "") or None
    if tz is None:
        default = source.default
        # И такой зоны нет - берем временную зону по умолчанию
        tz = default.tz
    return tz


def get_active_timezones(source_for_quote, quote):
    """
    Возвращает активную временную зону для котировки quote.
    """
    assert isinstance(quote, Quote), "Expected Quote, received %s." % quote.__class__.__name__
    assert isinstance(source_for_quote, Source), "Expected source object for get_active_timezones."
    timezones = []
    # Пробегаемся по карте и ищем нашу котировку.
    for map_name, map_ in source_for_quote.parser.maps.items():
        for from_, quote_info in map_.items():
            map_quote = quote_info["quote"]
            if map_quote is not None and map_quote.quote_id == quote.quote_id:
                if quote_info["tz"] is not None:
                    # Нашли котировку в карте и для нее установлена особенная временная зона.
                    timezones.append((map_name, map_, quote_info["tz"]))
                else:
                    # Спрашиваем временную зону у парсера.
                    timezones.append((map_name, map_, get_timezone(source_for_quote.parser, source_for_quote)))
    return timezones
