# -*- coding: utf-8 -*-

"""
Транспорт для данных Finance.ua.
"""

__author__ = "Zasimov Alexey"
__email__ = "zasimov-a@yandex-team.ru"


from stocks3.transports.xmlrpc import XmlRpcTransport
from stocks3.core.place import Place
from stocks3.core.factories import transports, places


class SpecialFinanceUaPlace(Place):
    """
    Специальное место для хранения данных от Finance.ua.
    """

    def __init__(self, tree, node):
        Place.__init__(self, tree, node)
        self.regions_data = None
        self.ukraine_data = None

places.register("stocks3.places.FinanceUa", SpecialFinanceUaPlace)


class FinanceUaTransport(XmlRpcTransport):

    places = [SpecialFinanceUaPlace]

    def __init__(self, tree, node, source):
        XmlRpcTransport.__init__(self, tree, node, source)
        self.proxy = self.getProxy()

    def makeConfig(self):
        XmlRpcTransport.makeConfig(self)
        self.primary_key = self.readString("query", "primary-key")
        # Читаем список загружаемых котировок
        self.quotes = []
        for quote_node in self.node.findall("query/quotes/quote"):
            quote = quote_node.text.strip()
            self.quotes.append(quote)

    def getSecondaryKey(self):
        """
        Возвращает сессионный ключ. Первичный ключ читается из конфигурации.
        """
        answer_with_skey = self.proxy.Finance.Security.Get.SecondaryKey( {'auth_id': self.primary_key} )
        return answer_with_skey['auth_id']

    def _make_args(self, sk):
        args = {'auth_id': sk, 'currency': self.quotes}
        return args

    def getRegionsData(self, sk):
        """
        Получаем данные по регионам.

        @param sk: Сессионный ключ (смотрим L{FinanceUaTransport.getSecondaryKey})
        """
        args = self._make_args(sk)
        return self.proxy.Finance.Cash.GetAverage.Current.Regions(args)

    def getUkraineData(self, sk):
        """
        Получаем данные по Украине.

        @param sk: Сессионный ключ (смотрим L{FinanceUaTransport.getSecondaryKey})
        """
        args = self._make_args(sk)
        return self.proxy.Finance.Cash.GetAverage.Current.Ukraine(args)

    def transfer(self, place):
        XmlRpcTransport.transfer(self, place)
        secondaryKey = self.getSecondaryKey()
        place.regions_data = self.getRegionsData(secondaryKey)
        place.ukraine_data = self.getUkraineData(secondaryKey)


transports.register("stocks3.transports.FinanceUa", FinanceUaTransport)

