# -*- coding: utf-8 -*-

"""
Пакет используется в тестах Stocks3.

Очень удобные функции:
    - L{load} - загружает нужный объект, принимая на вход имя XML-файла.
"""

__author__ = "Zasimov Alexey"
__email__ = "zasimov-a@yandex-team.ru"


import stocks3
stocks3.load_modules()
from stocks3.core.config import parse_xml
from stocks3.core.default import Default
from stocks3.core import factories
from datetime import datetime


Q1_ID = 919191
Q2_ID = 929292

# Functions from other modules
today = datetime.today


factoriesD = {"source": factories.sources,
              "transport": factories.transports,
              "parser": factories.parsers,
              "calculator": factories.calculators,
              "checker": factories.checkers,
              "saver": factories.savers}


def load_default():
    """
    Загружает тестовый Default.
    """
    return Default("tests/config/default.xml")


def load(xmlFileName, *args):
    """
    Создает объект по конфигурационном файлу xmlFileName.

    @param xmlFileName: Путь до XML-файла с конфигурацией.
    """
    tree = parse_xml(xmlFileName)
    root = tree.getroot()

    factory = factoriesD[root.tag]
    className = root.attrib.get("class", "stocks3.sources.Standart")
    return factory.create(className, tree, root, *args)


def load_source():
    """
    Загружает наш любимый тестовый источник.
    """
    default = load_default()
    return load("tests/config/source.xml", default)


def get_q1(source):
    return source.parser.default_map.getQuoteObject('SQ1')


def get_q2(source):
    return source.parser.default_map.getQuoteObject('SQ2')


class TestS3:
    """
    Тестовое окружение.
    """
    def setUp(self):
        self.source = load_source()
        self.q1 = get_q1(self.source)
        self.q2 = get_q2(self.source)


class DatabaseWP:
    """
    База данных, которая умеет отдавать предыдущее значение котировки.
    """
    def __init__(self, values):
        """
        @param quote: Котировка.
        @param values: Список значений. Эти значения и будет отдавать база.
        """
        self.values = values
        self.index = 0

    def select_previous_value(self, source, price):
        if self.index == len(self.values):
            self.index = 0
        try:
            return price.quote.make_price(today(), self.values[self.index])
        finally:
            self.index = self.index + 1


class DatabaseWPDual(DatabaseWP):
    def select_previous_value(self, source, price):
        if self.index == len(self.values):
            self.index = 0
        try:
            v = self.values[self.index]
            return price.quote.make_dual_price(today(), v[0], v[1])
        finally:
            self.index = self.index + 1

