#!/usr/bin/env python3
# coding: utf-8
import sys
import os.path
from collections import defaultdict
import json
import codecs
from lxml import etree

sys.path.append(".")
from libs import elementflow

'''
<key id="1.name" is_plural="False">
    <value language="ru" status="approved">USD ЦБ РФ</value>
    <value language="en" status="approved">USD CB RF</value>
    <value language="be" status="approved">USD ЦБ РФ</value>
    <value language="kk" status="approved">USD ЦБ РФ</value>
    <value language="tt" status="approved">USD ЦБ РФ</value>
    <value language="uk" status="approved">USD ЦБ РФ</value>
</key>
<key id="1.news_title" is_plural="False">
    <value language="ru" status="approved">Динамика курса USD ЦБ РФ, руб.</value>
    <value language="en" status="approved">USD CB RF rate dynamics, RUR.</value>
    <value language="be" status="approved">Динамика курса USD ЦБ РФ, руб.</value>
    <value language="kk" status="approved">Динамика курса USD ЦБ РФ, руб.</value>
    <value language="tt" status="approved">Динамика курса USD ЦБ РФ, руб.</value>
    <value language="uk" status="approved">Динамика курса USD ЦБ РФ, руб.</value>
</key>
<key id="1.line.1" is_plural="False">
    <value language="ru" status="approved">продажа</value>
    ...
</key>
'''

'''
<stock id="1">
    <name lang="ru">USD ЦБ РФ<name>
    <name lang="en">USD CB RF</name>
    <name lang="be">USD ЦБ РФ</name>
    <name lang="kk">USD ЦБ РФ</name>
    <name lang="tt">USD ЦБ РФ</name>
    <name lang="uk">USD ЦБ РФ</name>
    <news_title lang="ru">Динамика курса USD ЦБ РФ, руб.</name>
    <news_title lang="en">USD CB RF rate dynamics, RUR.</name>
    <news_title lang="be">Динамика курса USD ЦБ РФ, руб.</name>
    <news_title lang="kk">Динамика курса USD ЦБ РФ, руб.</name>
    <news_title lang="tt">Динамика курса USD ЦБ РФ, руб.</name>
    <news_title lang="uk">Динамика курса USD ЦБ РФ, руб.</name>
    <line_title id="1" lang="uk">купiвля</line_title>
    <line_title id="2" lang="uk">продаж</line_title>
    <line_title id="1" lang="ru">покупка</line_title>
    <line_title id="2" lang="ru">продажа</line_title>
    <line_title id="1" lang="by">покупка</line_title>
    <line_title id="2" lang="by">продажа</line_title>
</stock>
'''


def mkfile(quote_id, subkeys, path):
    with open(path, 'wb') as translation_file:
        with elementflow.xml(translation_file, 'stock', indent=True, attrs={'id': quote_id}) as xml:
            for subkey, values in subkeys.items():
                for lang, value, node_id in values:
                    if value is None:
                        # print(quote_id, subkey, lang)
                        value = ''
                    attrs = {'lang': lang}
                    if node_id is not None:
                        attrs["id"] = str(node_id)
                    xml.element(subkey, text=value, attrs=attrs)


if sys.version_info < (2, 6):
    _u = lambda text: text if text is None else text.encode("utf-8")
else:
    _u = lambda text: text if text is None else str(text)


def test_stock_text(quote_id, subkey, text):
    if "\n" in text:
        print("WARNING: %s.%s contains nl symbol:" % (quote_id, subkey))
        # print(_u(text))


def transform(tanker, translation_dir):
    with codecs.open(tanker, 'r', 'utf-8') as tankerfile:
        tanker = etree.parse(tankerfile)

    # Группируем по id
    info = defaultdict(dict)
    for key in tanker.xpath('//key'):
        splitted = key.get('id').split('.')
        if len(splitted) == 2:
            item_id, subkey = splitted
            node_id = None
        else:
            item_id, subkey, node_id = splitted
        if item_id not in info:
            info[item_id] = {}
        if subkey not in info[item_id]:
            info[item_id][subkey] = []
        for value in key.findall('value'):
            text = value.text
            info[item_id][subkey].append((value.get('language'), text, node_id))

    # Пишем в файлы
    new_info = {}
    for item_id, subkeys in list(info.items()):
        new_subkeys = {}
        for subkey, values in list(subkeys.items()):
            new_subkeys[subkey] = dict(v[:2] for v in [v for v in values if v[2] is None or v[2] == '1'])
            secondary = dict(v[:2] for v in [v for v in values if v[2] == '2'])
            if len(secondary) > 0:
                new_subkeys[subkey + '_2'] = secondary
        new_info[item_id] = new_subkeys
    with codecs.open(os.path.join(translation_dir, '../index.json'), 'w', 'utf-8') as f:
        data = json.dumps(new_info, ensure_ascii=False)
        f.write(data)

    for item_id, subkeys in list(info.items()):
        path = os.path.join(translation_dir, '%s.xml' % item_id)
        mkfile(item_id, subkeys, path)


def main():
    try:
        progname, tanker, translation_dir = sys.argv
    except ValueError:
        sys.stderr.write('usage: %s <tanker_file> <translation_dir>\n' % sys.argv[0])
        sys.exit(1)

    transform(tanker, translation_dir)


if __name__ == '__main__':
    main()
