# -*- coding: utf-8 -*-

"""
Скрипт, запускающий процесс обработки данных.

"""
import logging
import json
from datetime import datetime, timedelta
from pymongo import MongoClient, UpdateOne
from pprint import pprint
import sys
sys.path.append(".")
from stocks3.share.curl import second_load_url_data
from libs import xmltodict
from stocks3.core.common import get_field


url_template = "https://cbar.az/currencies/{date}.xml"


quote_map = {
    "USD": 76000,
    "EUR": 76001,
    "RUB": 76002,
    "ARS": 76003,
    "BYN": 76004,
    "BRL": 76005,
    "AED": 76006,
    "ZAR": 76007,
    "KRW": 76008,
    "CZK": 76009,
    "CLP": 76010,
    "CNY": 76011,
    "DKK": 76012,
    "GEL": 76013,
    "HKD": 76014,
    "INR": 76015,
    "GBP": 76016,
    "IDR": 76017,
    "IRR": 76018,
    "SEK": 76019,
    "CHF": 76020,
    "ILS": 76021,
    "CAD": 76022,
    "KWD": 76023,
    "KZT": 76024,
    "KGS": 76025,
    "LBP": 76026,
    "MYR": 76027,
    "MXN": 76028,
    "MDL": 76029,
    "EGP": 76030,
    "NOK": 76031,
    "UZS": 76032,
    "PLN": 76033,
    "AUD": 76034,
    "SGD": 76035,
    "SAR": 76036,
    "SDR": 76037,
    "TRY": 76038,
    "TWD": 76039,
    "TJS": 76040,
    "TMT": 76041,
    "UAH": 76042,
    "JPY": 76043,
    "NZD": 76044,
}


processed_dates = set()


class Database:
    def __init__(self, connectionstring):
        self.client = MongoClient(connectionstring, readPreference='secondary')
        self.stocks = self.client.stocksdb.stocks


class FakeSource:
    weight = 50


def make_database_write(saver, query, on_date):
    if len(query) > 0:
        try:
            res = saver.stocks.bulk_write(query)
            print("date:", on_date.strftime("%Y-%m-%d"), "modified_count:", res.modified_count, "upserted_count:", res.upserted_count)
        except Exception as e:
            print("date:", on_date.strftime("%Y-%m-%d"), "insert error:", e)
    else:
        print("date:", on_date.strftime("%Y-%m-%d"), "skip inserts")


def process(source, saver, logger):
    counter = 0
    query = []

    date_from = datetime.today()
    date_to = datetime.strptime("01.01.2006", "%d.%m.%Y")
    # date_to = datetime.strptime("01.03.2022", "%d.%m.%Y")

    on_date = date_from
    while on_date > date_to:
        date_str = on_date.strftime("%d.%m.%Y")

        url = url_template.format(date=date_str)

        data = second_load_url_data(url)
        data = xmltodict.parse(data)

        fakt_date = get_field(data, 'ValCurs.@Date', False)
        if fakt_date not in processed_dates:
            quotes_array = get_field(data, 'ValCurs.ValType.1.Valute', True, default=[])

            for quote in quotes_array:
                quote_value = quote['Value']
                quote_value = quote_value.replace(",", ".")
                print(quote['@Code'], quote_map[quote['@Code']], fakt_date, quote_value)
                db_date = datetime.strptime(fakt_date, "%d.%m.%Y")

                stock_value = {
                    '$set': {
                        'unixtime': int(on_date.timestamp()),
                        'time': '00:00:00',
                        'value': quote_value,
                        'lastupdate': datetime.now(),
                    },
                    '$setOnInsert': {
                        'id': quote_map[quote['@Code']],
                        'source': source.weight,
                        'region': 0,
                        'date': db_date.strftime("%Y-%m-%d"),
                    },
                }
                find_conditions = {
                    'id': quote_map[quote['@Code']],
                    'region': 0,
                    'date': db_date.strftime("%Y-%m-%d"),
                }
                if quote['@Code'] in quote_map:
                    query.append(UpdateOne(find_conditions, stock_value, upsert=True))
                else:
                    print("warn:", quote['@Code'], "not found")

        counter += 1
        if counter % 100 == 0:
            make_database_write(saver, query, on_date)
            query = []

        processed_dates.add(fakt_date)
        on_date -= timedelta(days=1)
    make_database_write(saver, query, on_date)


def main():
    with open("config/config.json") as c:
        config = json.load(c)
        database = config["db"]["connectionstring"]
    logging.basicConfig(level=logging.WARNING)
    logger = logging.getLogger('IMPORT')
    saver = Database(database)
    source = FakeSource()
    process(source, saver, logger)


if __name__ == "__main__":
    main()

