# -*- coding: utf-8 -*-

"""
Скрипт, запускающий процесс обработки данных.

"""
import logging
import json
from datetime import datetime, timedelta
from pymongo import MongoClient, UpdateOne
from pprint import pprint
import sys
sys.path.append(".")
from stocks3.share.curl import second_load_url_data
from libs import xmltodict
from stocks3.core.common import get_field


url = "https://nbg.gov.ge/gw/api/ct/monetarypolicy/currencies/en/json/?date={}"
quote_map = {
    "AED": 78000,
    "AMD": 78001,
    "AUD": 78002,
    "AZN": 78003,
    "BGN": 78004,
    "BRL": 78005,
    "BYN": 78006,
    "CAD": 78007,
    "CHF": 78008,
    "CNY": 78009,
    "CZK": 78010,
    "DKK": 78011,
    "EGP": 78012,
    "EUR": 78013,
    "GBP": 78014,
    "HKD": 78015,
    "HUF": 78016,
    "ILS": 78017,
    "INR": 78018,
    "IRR": 78019,
    "ISK": 78020,
    "JPY": 78021,
    "KGS": 78022,
    "KRW": 78023,
    "KWD": 78024,
    "KZT": 78025,
    "MDL": 78026,
    "NOK": 78027,
    "NZD": 78028,
    "PLN": 78029,
    "QAR": 78030,
    "RON": 78031,
    "RSD": 78032,
    "RUB": 78033,
    "SEK": 78034,
    "SGD": 78035,
    "TJS": 78036,
    "TMT": 78037,
    "TRY": 78038,
    "UAH": 78039,
    "USD": 78040,
    "UZS": 78041,
    "ZAR": 78042,
}


class Database:
    def __init__(self, connectionstring):
        self.client = MongoClient(connectionstring, readPreference='secondary')
        self.stocks = self.client.stocksdb.stocks


class FakeSource:
    weight = 50


def make_database_write(saver, query, on_date):
    if len(query) > 0:
        try:
            res = saver.stocks.bulk_write(query)
            print("date:", on_date.strftime("%Y-%m-%d"), "modified_count:", res.modified_count, "upserted_count:", res.upserted_count)
        except Exception as e:
            print("date:", on_date.strftime("%Y-%m-%d"), "insert error:", e)
    else:
        print("date:", on_date.strftime("%Y-%m-%d"), "skip inserts")


def process(source, saver, logger):
    counter = 0
    query = []

    date_from = datetime.today()
    date_from -= timedelta(days=1)
    date_to = datetime.strptime("2021-01-01", "%Y-%m-%d")
    # date_to = datetime.strptime("2022-07-20", "%Y-%m-%d")

    on_date = date_from
    while on_date > date_to:
        date_str = on_date.strftime("%Y-%m-%d")
        exact_url = url.format(date_str)

        data = second_load_url_data(exact_url)
        data = json.loads(data)

        if len(data) == 0:
            print('skip empty data', date_str)
            continue
        if 'currencies' not in data[0]:
            print('skip empty data currencies', date_str)
            continue

        for quote in data[0]['currencies']:
            # print(date_str, quote['code'], quote_map[quote['code']], quote['rate'], 'quantity:', quote['quantity'],
            #       round(quote['rate']/quote['quantity'], 8))
            stock_value = {
                '$set': {
                    'unixtime': int(on_date.timestamp()),
                    'time': '00:00:00',
                    'value': round(quote['rate']/quote['quantity'], 8),
                    'lastupdate': datetime.now(),
                },
                '$setOnInsert': {
                    'id': quote_map[quote['code']],
                    'source': source.weight,
                    'region': 0,
                    'date': date_str,
                },
            }
            find_conditions = {
                'id': quote_map[quote['code']],
                'region': 0,
                'date': date_str,
            }
            if quote['code'] in quote_map:
                query.append(UpdateOne(find_conditions, stock_value, upsert=True))
            else:
                print("warn:", quote['code'], "not found")

        counter += 1
        if counter % 100 == 0:
            make_database_write(saver, query, on_date)
            query = []
        on_date -= timedelta(days=1)
    make_database_write(saver, query, on_date)


def main():
    with open("config/config.json") as c:
        config = json.load(c)
        database = config["db"]["connectionstring"]
        # database = "prodmongo"
    logging.basicConfig(level=logging.WARNING)
    logger = logging.getLogger('IMPORT')
    saver = Database(database)
    source = FakeSource()
    process(source, saver, logger)


if __name__ == "__main__":
    main()

