#!/usr/bin/python
#  -*- coding: utf-8 -*-
#
# Copyright 2015 Google Inc. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""This example retrieves keywords that are related to a given keyword.
The LoadFromStorage method is pulling credentials and properties from a
"googleads.yaml" file. By default, it looks for this file in your home
directory. For more information, see the "Caching authentication information"
section of our README.
Tags: TargetingIdeaService.get
Api: AdWordsOnly
"""
from __future__ import division

__author__ = ('api.kwinter@gmail.com (Kevin Winter)'
              'Joseph DiLallo'
              'aalogachev@yandex-team.ru (Alexander Logachev)')

from googleads import adwords
import logging, os, ConfigParser
from ru.yandex import utils
import sys
reload(sys)
sys.setdefaultencoding("utf-8")

logger = logging.getLogger(__name__)

def get_queries_stat(queries, client, mode='STATS', stat_file=None):
    res = dict() #key - keyword, value - dict (month_volume, avg_cpc, compatition, categories)

    if stat_file is not None:
        out_file = open(stat_file, mode = 'w')
        out_file.write(u'{}\t{}\t{}\t{}\t{}\n'.format('keyword', 'month_volume', 'avg_cpc', 'competition', 'categories'))
    else:
        out_file = None

    # Initialize appropriate service.
    targeting_idea_service = client.GetService(
        'TargetingIdeaService', version='v201506')

    PAGE_SIZE = 300
    #FIXME check formula for paging
    for i in range(0, int(len(queries)/PAGE_SIZE)+1):
        logger.info('Processing page {0} of {1} [max {2} queries]'.format(i+1, int(len(queries)/PAGE_SIZE+1), PAGE_SIZE))
        # Construct selector object and retrieve related keywords.
        offset = 0
        selector = {
            'searchParameters': [
                {
                    'xsi_type': 'RelatedToQuerySearchParameter',
                    'queries': queries[i*PAGE_SIZE:(i+1)*PAGE_SIZE]
                },
                {
                    # Language setting (optional).
                    # The ID can be found in the documentation:
                    #  https://developers.google.com/adwords/api/docs/appendix/languagecodes
                    # downloaded version located in project root - languagecodes.csv
                    'xsi_type': 'LanguageSearchParameter',
                    'languages': [{
                        # 'id': '1000' - English
                        'id': '1031'  # - Russian
                        # 'id' : '1037' #- Turkish
                    }]
                }
            ],
            'ideaType': 'KEYWORD',
            'requestType': mode,
            # Product and Services categories - https://developers.google.com/adwords/api/docs/appendix/productsservices
            # downloaded version located in project root -
            'requestedAttributeTypes': ['KEYWORD_TEXT', 'SEARCH_VOLUME',
                                        'AVERAGE_CPC', 'COMPETITION', 'CATEGORY_PRODUCTS_AND_SERVICES'],
            'paging': {
                'startIndex': str(offset),
                'numberResults': str(PAGE_SIZE)
            }
        }

        more_pages = True
        while more_pages:
            page = targeting_idea_service.get(selector)

            # Display results.
            if 'entries' in page:
                for result in page['entries']:
                    attributes = {}
                    for attribute in result['data']:
                        attributes[attribute['key']] = getattr(attribute['value'], 'value', '0')
                    avg_cpc = 0
                    (keyword, month_volume, competition, categories) = (
                    attributes['KEYWORD_TEXT'], attributes['SEARCH_VOLUME'], attributes['COMPETITION'],
                    attributes['CATEGORY_PRODUCTS_AND_SERVICES'])
                    if (attributes['AVERAGE_CPC'] != '0'):
                        avg_cpc = attributes['AVERAGE_CPC'].microAmount / 1000000
                    res[keyword] = {'month_volume': month_volume, 'avg_cpc':avg_cpc, 'competition': competition, 'categories':categories}
                    if (out_file is not None):
                        out_file.write(u'{}\t{}\t{}\t{}\t{}\n'.format(keyword, month_volume, avg_cpc, competition, categories))
            else:
                logger.info('No keywords statistics')
            offset += PAGE_SIZE
            selector['paging']['startIndex'] = str(offset)
            more_pages = offset < int(page['totalNumEntries'])
    if out_file is not None:
        out_file.close()
    return res

def get_cluster_queries(file_name, pos):
    res = []
    fin = open(file_name, mode = 'r')
    for line in fin:
        res.append(line.split('\t', pos+1)[pos])
    fin.close()
    return res


if __name__ == '__main__':
    logging.basicConfig(format='%(asctime)s : %(levelname)s : %(message)s', level=logging.DEBUG)
    for handler in logging.root.handlers:
        handler.addFilter(logging.Filter('ru.yandex'))

    config = ConfigParser.SafeConfigParser()
    config.readfp(open(os.path.join(utils.get_project_path(), 'config.json')))
    section = config.defaults()['active_section']
    logger.debug('Using config section =' + section)
    work_dir = config.get(section=section, option='work_dir')
    logger.debug("Using work dir = " + work_dir)
    queries_file = config.get(section, 'queries_file')
    stat_file = config.get(section, 'google_stat_file')

    # Initialize client object.
    adwords_client = adwords.AdWordsClient.LoadFromStorage()
    queries = get_cluster_queries(queries_file, 0)
    logger.debug('Loaded {0} queries'.format(len(queries)))
    queries_stat = get_queries_stat(queries, adwords_client, 'STATS', stat_file)
    logger.debug('Get {0} queries stats'.format(len(queries_stat)))
    if logger.isEnabledFor(logging.DEBUG):
        for keyword, stat in queries_stat.items():
            logger.debug(u'{0} = {1}'.format(keyword, stat))
