import datetime
from optparse import OptionParser, OptionError

import reports_util


USAGE = '''
%script.py --country=ru --daydiff=1 --dayfrom=YYYYMMDD --dayto=YYYYMMDD
    --country               ru/tr
    --daydiff               upload data for n-th day back (0 for today, 1 for yesterday)
    --dayfrom, --dayto      update report for specific period, format YYYYMMDD
    --days                  update report for specific list of days, format YYYYMMDD,YYYYMMDD...
    --mute                  mute all exceptions
    --retries               number of retries (if mute enabled)
'''


def parse_options(update_report_for_day):
    def update_report_for_day_list(days_list, options):
        failed_days_list = []
        for day in days_list[::-1]:
            try:
                update_report_for_day(day, options)
            except Exception as e:
                if options.mute:
                    print(e)
                    failed_days_list.append(day)
                else:
                    raise
        return failed_days_list

    def parse_options_decorator():
        parser = OptionParser(USAGE)
        parser.add_option('--country', action='store', dest='country', default=None,
                          help='Trigger update report for the specific country (ru/tr are currently supported')
        parser.add_option('--daydiff', action='store', dest='daydiff', default=None,
                          help='Trigger update report for the n-th previous day')
        parser.add_option('--dayfrom', action='store', dest='dayfrom', default=None,
                          help='Update report for a specific period. Format YYYYMMDD.')
        parser.add_option('--dayto', action='store', dest='dayto', default=None,
                          help='Update report for a specific period. Format YYYYMMDD.')
        parser.add_option('--days', action='store', dest='days', default=None,
                          help='Update report for a specific list of days. Format YYYYMMDD,YYYYMMDD,...')
        parser.add_option('--mute', action='store', dest='mute', default=None,
                          help='Mute all exceptions. Failed days will be printed at the end')
        parser.add_option('--retries', action='store', dest='retries', default=2,
                          help='If exceptions muted, retry it this number of times')
        (options, args) = parser.parse_args()

        days_list = []
        if options.daydiff is not None:
            nthday = (datetime.date.today() - datetime.timedelta(int(options.daydiff))).strftime('%Y-%m-%d')
            days_list.append(nthday)

        if options.dayfrom and options.dayto:
            days_list.extend(reports_util.days_range(options.dayfrom, options.dayto))

        if options.days:
            days_list.extend(reports_util.days_list(options.days))

        if not days_list:
            raise OptionError('No days to update. One of these options should be set: daydiff, dayfrom, dayto, days',
                              options)

        for i in range(int(options.retries)):
            days_list = update_report_for_day_list(days_list, options)

        if len(days_list) > 0:
            print('Finally failed days : {}'.format(days_list))

    return parse_options_decorator
