from collections import defaultdict, namedtuple
import ranker
import datetime
import cPickle
import os

class Player(object):
	def __init__(self, ui):
		# activity: date -> searches (e.g. 20151005 -> 4)
		self.activity = defaultdict(int)
		self.ui = ui
		self.yandexuids = set()
		self.installdate = None

	def log_install(self, date):
		self.installdate = date


def parse_date(yymmdd):
	return datetime.datetime.strptime(str(yymmdd), '%Y%m%d').date()
def unparse_date(date):
	return int(date.strftime('%Y%m%d'))

def week_until(date):
	"""
	>>> list(week_until(20160103))
	[20151228, 20151229, 20151230, 20151231, 20160101, 20160102, 20160103]
	"""

	d = parse_date(date)
	for td in reversed(range(7)):
		yield unparse_date(d - datetime.timedelta(days=td))

class Team(object):
	def __init__(self, banerid):
		self.banerid = banerid
		self.players = {} # ui -> Player (e.g. '2DC8AF1-...-B001' -> Player object)
		self.installs = defaultdict(set)
		self.total_installs = set()

	def add_install(self, date, player):
		self.players[player.ui] = player
		player.log_install(date)
		self.installs[date].add(player.ui)
		self.total_installs.add(player.ui)

	def get_weekly_stats(self, date):
		all_installs = reduce(set.union,
				(self.installs.get(d, set()) for d in week_until(date)))

		return {'installs': len(all_installs), 'active': 0, 'superactive': 0}

	def get_total_stats(self, date):
		return {'installs': len(self.total_installs), 'active': 0, 'superactive': 0}

class Game(object):
	def __init__(self):
		self.players = {} # 'ui' -> Player
		self.teams = {} # 'banerid' -> Team
		self.last_saved = None

	def add_install(self, date, banerid, ui):
		try:
			team = self.teams[banerid]
		except KeyError:
			team = Team(banerid)
			self.teams[banerid] = team
		try:
			player = self.players[ui]
		except KeyError:
			player = Player(ui)
			self.players[ui] = player
		team.add_install(date, player)

	def process_export_record(self, rec):
		stat = rec['stat']
		if stat == 'install':
			banerid = rec['banerid']
			if not banerid.startswith('60'):
				return
			ui = rec['ui']
			if len(ui) < 5:
				return
			try:
				ts = unparse_date(datetime.datetime.strptime(rec['ts'], '%FT%T'))
			except:
				ts = unparse_date(datetime.date.today())
			self.add_install(ts, banerid, ui)
		elif stat == 'uninstall':
			print 'uninstall not handled yet'

	def publish_scoreboard(self, date, storage):
		team_dict = dict((t.banerid,
			{'total': t.get_total_stats(date), 'week': t.get_weekly_stats(date)})
			for t in self.teams.itervalues())
		docs = ranker.rank(team_dict)
		for (url, doc) in docs:
			storage[url] = doc

	def maybe_save(self, save_dir):
		now = datetime.datetime.now()
		if self.last_saved and (now - self.last_saved < datetime.timedelta(hours=1)):
			return None
		dumpfile = os.path.join(save_dir, now.strftime('%FT%T'))
		self.last_saved = now
		cPickle.dump(self, file(dumpfile, 'w'), protocol=2)
		temp_link = os.path.join(save_dir, 'tmp')
		try:
			os.unlink(temp_link)
		except OSError: pass
		os.symlink(os.path.abspath(dumpfile), temp_link)
		os.rename(temp_link, os.path.join(save_dir, 'latest'))
		return dumpfile
