#include <mapreduce/interface/all.h>
#include <mapreduce/lib/init.h>
#include <mapreduce/library/io/stream/stream.h>
#include <mapreduce/library/temptable/temptable.h>

#include <library/cpp/map_text_file/map_tsv_file.h>

#include <util/datetime/systime.h>
#include <util/generic/set.h>
#include <library/cpp/cgiparam/cgiparam.h>
#include <util/string/strip.h>

static const TSet<TString> testBanerids = {
    "0101910664",
    "0101910627",
    "8186690025",
    "0101040674",
    "0201540200"
};

using namespace NMR;

using TDsv = THashMap<TStringBuf, TStringBuf>;

TDsv ParseDsv(TValue value) {
    TStringBuf v = value.AsStringBuf(), tok;
    TDsv result;
    while (v.NextTok('\t', tok)) {
        TStringBuf left, right;
        tok.Split('=', left, right);
        result[left] = right;
    }
    return result;
}

static time_t ParseTimestamp(const TString& ts) {
    struct tm t;
    if (strptime(ts.data(), "%d/%b/%Y:%H:%M:%S", &t) || strptime(ts.data(), "%Y-%m-%dT%H:%M:%S", &t)) {
        return mktime(&t);
    }
    return 0;
}

class TExportLogParser: public IMap {
    OBJECT_METHODS(TExportLogParser);

    THashSet<TString> KnownUIs;

public:
    SAVELOAD_OVERRIDE(IMap, KnownUIs);

    TExportLogParser() {}
    TExportLogParser(TTable& uiTable) {
        for (auto uis = uiTable.Begin(); uis.IsValid(); ++uis)
            KnownUIs.insert(uis.GetKey().AsString());
    }

    void Do(TValue /*key*/, TValue value, TUpdate &output) override {
        TDsv row = ParseDsv(value);
        TCgiParameters c;
        c.ScanAddAll(row["request"]);
        const TString& banerid = c.Get("banerid");
        bool goodBanerid = banerid.StartsWith("60") || testBanerids.contains(banerid);
        const TString& stat = c.Get("stat");
        const TString& ui = StripString(c.Get("ui"), [](const char* c) { return *c == '{' || *c == '}'; });
        bool interestingRow = (
            c.Get("yasoft") == "yabrowser" &&
            EqualToOneOf(stat, "install", "uninstall") &&
            (goodBanerid || KnownUIs.contains(ui)));
        if (!interestingRow)
            return;

        KnownUIs.insert(ui); // on the off chance that we encounter it again in the same job

        TValueOutput so, vo;
        output.Add(ui, vo << ParseTimestamp(TString(row["timestamp"])) << '\t'
                << stat << '\t' << banerid);
    }
};
REGISTER_SAVELOAD_CLASS(0x1235, TExportLogParser);

using TStringMap = THashMap<TString, TString>;

class TRequestCounter: public IReduce {
    OBJECT_METHODS(TRequestCounter);

    TStringMap YandexuidToUI; // yandexuid without "y" -> UI without {braces}

    SAVELOAD_OVERRIDE(IReduce, YandexuidToUI);

    TRequestCounter() {
    }

public:
    TRequestCounter(TStringMap y2u)
        : YandexuidToUI(y2u)
    {
    }

    void Do(TValue key, TTableIterator &input, TUpdate &output) override {
        if (!key.AsStringBuf().StartsWith('y'))
            return;
        TString* ui = YandexuidToUI.FindPtr(key.AsStringBuf().SubStr(1));
        if (!ui)
            return;
        size_t count = 0;
        for (; input.IsValid(); ++input) {
            if (input.GetValue().AsStringBuf().StartsWith("type=REQUEST"))
                count += 1;
        }
        if (count) {
            TValueOutput vo;
            output.Add(*ui, vo << count);
        }
    }
};
REGISTER_SAVELOAD_CLASS(0x1234, TRequestCounter);

class TDumbKeyCount: public IMap {
    OBJECT_METHODS(TDumbKeyCount);
    THashMap<TString, size_t> Counts;
    void Do(TValue key, TValue, TUpdate&) override {
        Counts[key.AsString()] += 1;
    }
    void Finish(ui32, TUpdate& update) override {
        for (auto kv : Counts) {
            TValueOutput vo;
            update.Add(kv.first, vo << kv.second);
        }
    }
};
REGISTER_SAVELOAD_CLASS(1445867119, TDumbKeyCount);

class TValueSum: public IReduce {
    OBJECT_METHODS(TValueSum);
    void Do(TValue key, TTableIterator& iter, TUpdate& output) override {
        size_t count = 0;
        for (; iter.IsValid(); ++iter) {
            count += FromString<size_t>(iter.GetValue().AsString());
        }
        TValueOutput vo;
        output.Add(key, vo << count);
    }
};
REGISTER_SAVELOAD_CLASS(1445867536, TValueSum);

int main(int argc, const char* argv[]) {
    Initialize(argc, argv);
    if (argv[1] == "parsets"sv) {
        Cout << ParseTimestamp(argv[2]) << Endl;
        return 0;
    }
    TServer server(argv[1]);
    if (argv[4] == "countkeys"sv) {
        server.MapReduce(argv[2], argv[3], new TDumbKeyCount, new TValueSum);
        server.Sort(argv[3]);
        return 0;
    }
    if (argv[4] == "export-log"sv) {
        TClient client(server);
        TTable uiTable(client, "/tmp/myltsev/uis");
        TExportLogParser* elp = new TExportLogParser(uiTable);
        server.Map(argv[2], argv[3], elp);
        server.Sort(argv[3]);
        return 0;
    }
    WithUniqBTTable tmp(server, "ui-counts");
    TStringMap y2u;
    for (auto tabs: TMapTsvFile(argv[4])) {
        y2u[tabs.at(0)] = tabs.at(1);
    }
    server.Reduce(argv[2], tmp.Name(), new TRequestCounter(std::move(y2u)));
    return 0;
}
