from django.core.mail import send_mail
from django.conf import settings
from django.utils import timezone
from .models import Snapshot, Entry, GlobalVar, Deployment
import subprocess
import datetime
import traceback
import requests
import json
from six import BytesIO
import tempfile
import shutil
import time
import os
import re
import os.path
import zipfile


def log_tb():
    f = open('/home/atom_admin/app/errors.log', 'a')
    f.write(traceback.format_exc())
    f.close()


def dump_entries_to(stream):
    entries = Entry.objects.all()
    stats = {'lists': {}}
    for e in entries:
        candidates = e.candidates()
        flat_json = e.flat_json().encode('utf-8')
        stream.write(b'\t'.join((
            b'#query', e.slug.encode('utf-8'), flat_json + b'\n')))
        avg_filter_count = 'NA' if not candidates else (
            sum(c.get('filter', '').count("'#") for c in candidates if isinstance(c, dict)) / len(candidates))
        stats['lists'][e.slug] = {
            'bytesize': len(flat_json),
            'candidate_count': len(candidates),
            'avg_filter_count': avg_filter_count,
        }
    stats['list_count'] = len(entries)
    return stats


def create_snapshot(comment, author):
    stats = {'comment': comment}
    with open('snapshot.txt', 'wb') as s:
        stats.update(dump_entries_to(s))
    now_ts = datetime.datetime.now().strftime('%s')
    zipfilename = '/mnt/backup/snapshot{}.zip'.format(
        now_ts
    )
    with zipfile.ZipFile(
        zipfilename, 'w', compression=zipfile.ZIP_DEFLATED
    ) as zf:
        zf.write('snapshot.txt')
    os.remove('snapshot.txt')
    return Snapshot.objects.create(
        text=zipfilename,
        comment=json.dumps(
            stats, ensure_ascii=False
        ),
        created=timezone.now(),
        author=author, trie_name=now_ts
    )


def create_deployment(login, comment):
    snapshot = create_snapshot(comment, login)
    return Deployment.objects.create(
        created=timezone.now(),
        author=login, comment=comment, state='new', snapshot=snapshot)


def sanitize_filename(n):
    return re.sub('[^-.a-zA-Z0-9/]', '_', n)


class SandboxFormat:

    def __init__(self, rootdir):
        self.rootdir = rootdir

    def write_qs(self, all_keys, trie_name):
        pass

    def write_version(self, version):
        directory, filename = self.get_dir_and_file('VERSION')
        with open(os.path.join(directory, filename), 'wb') as f:
            f.write(version)

    def get_dir_and_file(self, path):
        directory, _, filename = sanitize_filename(path).rpartition('/')
        directory = os.path.join(self.rootdir, directory)
        return directory, filename

    def write_kv(self, key, values, mode='default'):
        if mode == 'LOAD_CONFIG':
            string = json.dumps(
                values, sort_keys=True, indent=2
            ).encode('utf-8')
        elif mode == 'default':
            string = (b'[' + b','.join(
                json.dumps(v, sort_keys=True, indent=2).encode('utf-8')
                for v in values
            ) + b']\n')
        avg_filter_count = 'NA' if not values else (
            string.count(b"'#") * 1.0 / len(values))
        stats = {
            'bytesize': len(string),
            'candidate_count': len(values),
            'avg_filter_count': avg_filter_count,
        }
        directory, filename = self.get_dir_and_file(key)
        if not os.path.isdir(directory):
            os.makedirs(directory)
        with open(
            os.path.join(directory, get_sandbox_name(filename)), 'wb'
        ) as f:
            f.write(string)
        return stats


def get_sandbox_name(filename):
    if filename == 'LOAD_CONFIG':
        return filename
    return filename + '.json'


class SandboxApi:
    root_url = 'https://sandbox.yandex-team.ru/api/v1.0/'

    def __init__(self, token):
        self.token = token

    @staticmethod
    def retry(request, *args, **kwargs):
        for attempt in range(5, 0, -1):
            try:
                return request(*args, **kwargs)
            except requests.ConnectionError:
                log_tb()
                if attempt == 1:
                    raise
                else:
                    time.sleep(1)

    def post(self, url, **kwargs):
        r = self.retry(requests.post, self.root_url + url, headers={
            'Authorization': 'Bearer ' + self.token}, **kwargs)
        if not str(r.status_code).startswith('2'):
            raise Exception('{}: {}'.format(r.status_code, r.content))
        return r

    def put(self, url, **kwargs):
        r = self.retry(requests.put, self.root_url + url, headers={
            'Authorization': 'Bearer ' + self.token}, **kwargs)
        if not str(r.status_code).startswith('2'):
            raise Exception('{}: {}'.format(r.status_code, r.content))
        return r

    def get(self, url, **kwargs):
        return self.retry(requests.get, self.root_url + url, **kwargs)

    def create_task(self, task_type, context):
        return self.post('task', json={'type': task_type, 'author': 'robot-atom-banner', 'context': context}).json()

    def update_task(self, task_id, values):
        self.put('task/{}'.format(task_id), json=values)

    def run_task(self, task_id):
        return self.put('batch/tasks/start', json=[task_id]).json()

    def increase_priority(self, task_id, N):
        for i in range(N):
            self.put('batch/tasks/increase_priority', json=[task_id])

    def get_status(self, task_id):
        try:
            return self.get('task/{}'.format(task_id)).json()['status']
        except ValueError:
            return None

    def release(self, task_id, to, subject):
        self.post('release', json={"cc": [], "type": to,
                                   "task_id": task_id, "subject": subject, "message": ""})


def upload_to_sandbox(deployment, sb_token):
    snapshot = deployment.snapshot
    now_ts = datetime.datetime.now().strftime('%s')
    for path in os.listdir('.'):  # clean up old deployments
        if not os.path.isdir(path):
            continue
        try:
            then_ts = int(path)
        except ValueError:
            log_tb()
            continue
        if abs(int(now_ts) - then_ts) >= 3600:
            shutil.rmtree(path)
    cwd = os.getcwd()
    os.mkdir(now_ts)
    os.chdir(now_ts)
    candidates_dir = 'atom_candidates'
    candidates_tar = 'atom_candidates.tar.gz'
    if os.path.isdir(candidates_dir):
        shutil.rmtree(candidates_dir)
    if os.path.isfile(candidates_tar):
        os.remove(candidates_tar)
    os.mkdir(candidates_dir)
    os.chmod(candidates_dir, 0755)
    try:
        sandbox_pack = SandboxFormat(candidates_dir)
        stats = snapshot.write_tsv(sandbox_pack)
        subprocess.call(['tar', 'cvzf', candidates_tar,
                         candidates_dir])
        if not os.path.isfile(candidates_tar):
            raise Exception('atom_candidates.tar.gz is missing')
        os.chmod(candidates_tar, 0777)
        if True:  # the Sandbox task uses backbone now
            deployment.add_log('starting "sky share"')
            url = subprocess.check_output(['sky', 'share', '-Trbtorrent',
                                           candidates_tar]).strip()
            sb = SandboxApi(sb_token)
            deployment.add_log(
                'creating Sandbox task with url "{}"'.format(url))
            t = sb.create_task('DEPLOY_ATOM_CANDIDATES',
                               {'candidates_src': url})
            task_id = t['id']
            sb.update_task(task_id, {
                'description': 'Deploy Atom candidates.', 'owner': 'PERS'})
            deployment.add_log('starting task {}'.format(task_id))
            sb.run_task(task_id)
            sb.increase_priority(task_id, 4)
            last_status = None
            while True:
                status = sb.get_status(task_id)
                if status not in ['DRAFT', 'PREPARING', 'ENQUEUING', 'ENQUEUED', 'ASSIGNED',
                                  'EXECUTING', 'FINISHING', 'STOPPING', None]:
                    break
                if (status is not None) and (status != last_status):
                    deployment.add_log('task status: {}'.format(status))
                    last_status = status
                time.sleep(10)
            if status != 'SUCCESS':
                raise Exception('Task {}: status {}'.format(task_id, status))
            snapshot.task_id = task_id
            snapshot.save(update_fields=['task_id'])
            deployment.task_id = task_id
            deployment.save(update_fields=['task_id'])
            sb.release(task_id, to='stable', subject='task {} by {}'.format(
                task_id, deployment.author))
            return task_id, stats
        else:
            sandbox_pack.write_version(bytes(snapshot.pk))
            output = subprocess.check_output(['/home/myltsev/ya/ya',
                                              'upload', '--http', '--ttl=365', '--owner=robot-atom-banner',
                                              '-t', sb_token, '-T', 'PERS_ATOM_CANDIDATES', candidates_dir],
                                             stderr=subprocess.STDOUT)
            match = re.search('Created resource id is ([0-9]+)', output)
            resource_id = match.group(1) if match else 0
            return None
    finally:
        os.chdir(cwd)


class QuerysearchFormat:

    def __init__(self, f):
        self.f = f

    def write_qs(self, all_keys, trie_name):
        self.write_kv(b'all_keys', all_keys)
        self.write_kv(b'__trie_meta', [{'timestamp': trie_name}])

    def write_kv(self, key, values):
        string = b'\t'.join([
            b'#query',
            key,
            b'[' + b','.join(
                json.dumps(v, sort_keys=True, separators=(
                    ',', ':')).encode('utf-8')
                for v in values
            ) + b']\n'
        ])
        avg_filter_count = 'NA' if not values else (
            string.count(b"'#") * 1.0 / len(values))
        stats = {
            'bytesize': len(string),
            'candidate_count': len(values),
            'avg_filter_count': avg_filter_count,
        }
        self.f.write(string)
        return stats


def rollout(snapshot, login, testing=False):
    d = tempfile.mkdtemp(prefix='atom-admin-')
    stats = None
    try:
        f = tempfile.NamedTemporaryFile(dir=d)
        stats = snapshot.write_tsv(QuerysearchFormat(f))
        f.flush()
        subprocess.check_output([
            '/bin/bash', '-efx',
            os.path.join(settings.BASE_DIR,
                         'atom/update_atom_candidates_trie.sh'),
            d, f.name, snapshot.trie_name,
            'atom-test-candidates' if testing else 'atom-candidates'],
            stderr=subprocess.STDOUT)
    finally:
        f.close()
        shutil.rmtree(d)
    return stats


def get_qs_request(trie='atom-candidates'):
    url = 'http://querysearch-atom.search.yandex.net/yandsearch'
    qd_struct = {trie: {'__trie_meta': []}}
    params = dict(ms='querysearch:json:3', waitall='da', timeout='1000000',
                  rearr='qd_struct_keys={}'.format(json.dumps(qd_struct)))
    return requests.get(url, params=params)


def get_qs(trie='atom-candidates'):
    url = 'http://querysearch.search.yandex.net/yandsearch'
    entries = Entry.objects.all()
    candidates = dict((entry.slug, None) for entry in entries)
    qd_struct = {trie: candidates}
    params = dict(ms='querysearch:json:3', waitall='da', timeout='1000000',
                  rearr='qd_struct_keys={}'.format(json.dumps(qd_struct)))
    j = requests.get(url, params=params).json()
    timestamps = dict((d['Key'][0], d['Timestamp'])
                      for d in j['Data'] if 'Key' in d)
    ts_set = set(timestamps.get(e.slug, 0) for e in entries)
    if len(ts_set) == 1:
        return list(ts_set)[0]
    return 0


def send_notifications(snapshot, stats=None):
    recipients = set(('riddle', 'manokk', 'qkrorlqr', 'pecheny'))
    if snapshot.author and snapshot.author not in [
        'robot', 'robot@yandex-team.ru'
    ]:
        recipients.add(snapshot.author)
    send_mail(
        '[Atom] {login}: new trie {ts}'.format(
            login=(snapshot.author or 'unknown'), ts=snapshot.trie_name),
        '{stats}\nSnapshots: {url}'.format(
            url='https://switch.v.yandex-team.ru/atom/snapshots/',
            stats=snapshot.print_stats(stats)),
        'devnull@yandex-team.ru',
        ['{}@yandex-team.ru'.format(login) for login in recipients],
        fail_silently=False)


def can_freeze(login):
    return login in [
        'qkrorlqr', 'manokk', 'riddle',
        'librarian', 'kozhapenko', 'pecheny', 'chikachoff', 'zaringleb',
        'norberrt', 'const', 'ryabinin', 'zagrebin'
    ]


def set_freeze(setting, author):
    GlobalVar.set_value('freeze', (setting == 'on'), author)
