from django import forms
from six import string_types
from .models import Entry, BannerTemplate, CandidateUrl, AutoQuota
from .filters import valid_filter
from crispy_forms.helper import FormHelper
from crispy_forms.layout import Submit
import json

class JsonWidget(forms.Textarea):
    def render(self, name, value, attrs=None):
        if value:
            try:
                j = json.loads(value)
                value = json.dumps(j, ensure_ascii=False, indent=4)
            except: pass
        if not attrs:
            attrs = {}
        attrs['rows'] = 20
        return super(JsonWidget, self).render(name, value, attrs)

class JsonField(forms.Field):
    def __init__(self, widget=None, **kwargs):
        super(JsonField, self).__init__(widget=JsonWidget, **kwargs)
    def validate(self, value):
        try:
            json.loads(value)
        except:
            raise forms.ValidationError("Invalid JSON!")

class CandidateListField(forms.Field):
    def __init__(self, widget=None, **kwargs):
        super(CandidateListField, self).__init__(widget=JsonWidget, **kwargs)

    def validate(self, value):
        try:
            j = json.loads(value)
        except:
            raise forms.ValidationError("Invalid JSON!")
        if isinstance(j, list):
            for c in j:
                if isinstance(c, dict) and 'filter' in c:
                    f = c['filter']
                    if not isinstance(f, string_types):
                        raise forms.ValidationError("Filters must be strings.")
                    vf, ps = valid_filter(f)
                    if not vf:
                        raise forms.ValidationError("Invalid filter [%s], problem location: %s." % (f, ps))


class EntryForm(forms.ModelForm):
    json = CandidateListField()
    class Meta:
        model = Entry
        fields = ['slug', 'json']
        widgets = {'json': JsonWidget}

    def clean(self):
        super(EntryForm, self).clean()
        slug = self.cleaned_data.get('slug', '')
        if slug in {'all_keys', 'LOAD_CONFIG'}:
            return
        j = self.cleaned_data.get('json')
        if not j:
            return
        j = json.loads(j)
        if not isinstance(j, list):
            self.add_error('json', 'Expected a JSON list.')
            return
        urls = set()
        for c in j:
            if not isinstance(c, dict):
                continue
            u = c.get('internal-url')
            if not isinstance(u, string_types):
                self.add_error('json', 'Internal url not a string: {}'.format(u))
                return
            hostname, slash, bannerid = u.partition('/')
            if not (slash == '/' and hostname and bannerid):
                self.add_error('json',
                    'Invalid internal-url (must be HOSTNAME/BANNER_ID): [{}]' .format(u))
                return
            if bannerid in urls:
                self.add_error('json',
                    'Duplicate bannerid: [{}] (occurs in url: {})'.format(bannerid, u))
                return
            urls.add(bannerid)
        foreign_dups = CandidateUrl.objects.exclude(list__slug=slug).filter(url__in=urls)
        if foreign_dups:
            self.add_error('json', 'Banner ID collisions with other lists: {}'.format(
                ', '.join('{}@{}'.format(fdup.url, fdup.list.slug) for fdup in foreign_dups[:10])))

class BannerTemplateForm(forms.ModelForm):
    json = JsonField()
    class Meta:
        model = BannerTemplate
        fields = ['slug', 'json']

    @property
    def helper(self):
        h = FormHelper()
        h.add_input(Submit('save', 'Save'))
        return h


class AutoQuotaForm(forms.ModelForm):
    target_show_rate = forms.FloatField(min_value=0.01, max_value=1.0)

    class Meta:
        model = AutoQuota
        fields = ['production_list', 'target_show_rate', 'current_multiplier']

    @property
    def helper(self):
        h = FormHelper()
        h.add_input(Submit('save', 'Save'))
        return h
