from django.test import TestCase
from .models import BannerTemplate, Entry, Substitution
from .deploy import create_snapshot
from .views import add_texts_to_entry, extract_field_values, replace_values_in
import json

class SnapshotTests(TestCase):
    def test_stats(self):
        Entry(slug='dummy', json=json.dumps([{
            'filter': """'#referer,not_match,%2236032%'&'#referer,not_match,%2236033%'""",
            'url': 'search/dummy'}], ensure_ascii=False)).save()
        Entry(slug='empty', json="[]").save()
        s = create_snapshot('fix', 'myltsev')
        stats = json.loads(s.comment)
        self.assertEqual(stats, {'list_count': 2, 'comment': 'fix', 'lists': {
            'dummy': {'avg_filter_count': 2, 'bytesize': 100, 'candidate_count': 1},
            'empty': {'avg_filter_count': 'NA', 'bytesize': 2, 'candidate_count': 0}}})
        self.assertEqual(s.print_stats(stats),
            'Comment: fix\n\n2 lists:\n\n'
            '    dummy: 100 bytes, 1 candidates, 2 filters/candidate\n'
            '    empty: 2 bytes, 0 candidates, NA filters/candidate\n\n')

class TemplateTests(TestCase):
    def test_var_names(self):
        t = BannerTemplate(slug='test',
                json='''{"greeting": "{{hello}} {{world}}", "name": "{{name}}"}''')
        subs = t.get_variables([
            {"greeting": "Hello Earth", "name": "Julius"}])
        subs.sort()
        self.assertEqual(subs, [
            Substitution(u"hello", u''),
            Substitution(u"name", u"Julius"),
            Substitution(u"world", u'')])

    def test_var_substitution(self):
        t = BannerTemplate(slug='test',
                json='''{"greeting": "{{hello}} {{world}}", "name": "{{name}}"}''')
        values = {'hello': 'O, hi', 'world': 'planet "Earth"', 'name': 'O\'Hara'}
        self.assertEqual(json.loads(t.replace(values.items()).next()),
                {"greeting": 'O, hi planet "Earth"', "name": "O'Hara"})

    def test_value_substitution(self):
        t = BannerTemplate(slug='test',
                json='''{"weight": "{{weight:int}}", "wstr": "{{weight}}"}''')
        self.assertEqual(json.loads(t.replace({'weight': 200}).next()),
                {'weight': 200, 'wstr': '200'})
        self.assertEqual(json.loads(t.replace({'weight': '7'}).next()),
                {'weight': 7, 'wstr': '7'})
        self.assertEqual(json.loads(t.replace({'weight': 'foobar'}).next()),
                {'weight': 0, 'wstr': 'foobar'})

class BulkEditTests(TestCase):
    def test_edit_values(self):
        e = Entry(json='[{"url":"turk/bid000420", "weight":1}]')
        candidates = e.candidates()
        values = extract_field_values(None, candidates[0])
        repacements = {'weight': {'1': '2'}}
        replace_values_in(candidates, repacements)
        self.assertEqual(candidates, [{'url': 'turk/bid000420', 'weight': 2}])

