#!/usr/bin/env python
# -*- coding: utf-8 -*-
from __future__ import unicode_literals
from django.db import transaction
from django.shortcuts import render, get_object_or_404
from django.http import (HttpResponseRedirect, HttpResponse,
                         HttpResponseNotAllowed, JsonResponse)
from django.contrib import messages
from django.core.urlresolvers import reverse, reverse_lazy
from django.utils import timezone
from django.views.decorators.http import require_POST
from django.views.decorators.csrf import csrf_protect, csrf_exempt
from django.views.generic import ListView, CreateView, DeleteView, DetailView, UpdateView
import operator
import codecs
import os
import hashlib
import json
import re
import time
import traceback
import zipfile
import requests
import difflib
from . import diffs
import datetime
from six.moves import zip as izip, zip_longest as izip_longest
from six import text_type, string_types, python_2_unicode_compatible, iteritems
from itertools import product, repeat, chain
from collections import defaultdict

from . import deploy
from .filters import valid_filter
from .login import get_approved_login
from .models import (Entry, Lock, Editor, Log, Snapshot, BannerTemplate, GlobalVar,
                     EditLog, Deployment,
                     StatDatum, CandidateUrl,
                     get_list_sources_from_db,
                     AuthToken,
                     AutoQuota,
                     )
from .bannerid import get_n_bannerids, get_dups
from .forms import EntryForm, BannerTemplateForm, AutoQuotaForm
from .tasks import two_step_rollout


def get_recently_edited(login):
    month_ago = timezone.now() - datetime.timedelta(days=31)
    return set(e.list_slug for e in EditLog.objects.filter(
        login=login, timestamp__gt=month_ago).only('list_slug').distinct())


def get_entry_tags(all_keys):
    result = defaultdict(set)
    for m in all_keys:
        if isinstance(m, string_types):
            result[m].add(m)
            continue
        for entry, lists in iteritems(m):
            result[entry].update(lists)
    return result


def entries_for_login(login):
    editor = Editor.objects.filter(login=login)
    slugs = []  # empty filter -> all entries
    if editor:
        slugs = editor[0].get_allowed_slugs()
    if slugs:
        return Entry.objects.filter(slug__in=slugs)
    return Entry.objects.all()


def list_view(request):
    login = get_approved_login(request)
    entries = dict((e.slug, e) for e in entries_for_login(login).defer('json'))

    my_slugs = get_recently_edited(login)

    all_keys = entries.get('all_keys')
    all_keys = all_keys.candidates() if all_keys else []
    entry_tags = get_entry_tags(all_keys)

    for e in entries.values():
        e.mine = e.slug in my_slugs
        e.tags = sorted(entry_tags.get(e.slug, []))
        e.enabled = e.slug in e.tags

    entry_list = sorted(entries.values(), key=lambda e: (
        not e.mine, e.enabled, e.slug))
    return render(request, 'atom/entry_list.html', dict(
        entries=entry_list,
        locks=Lock.objects.all(),
        me=login,
        freeze=GlobalVar.get('freeze'), can_freeze=deploy.can_freeze(login)))


def visual_search(request, query=''):
    if not query:
        if 'query' in request.POST:
            query = request.POST['query']
        else:
            return JsonResponse({'error': 'empty search query'})
    search_function = find_candidate_by_fulltext
    if request.POST.get('fulltext_search') != 'on':
        search_function = find_candidate_by_id
    search_results = search_function(query)
    if 'error' in search_results:
        messages.add_message(
            request, messages.ERROR,
            '{} not found :('.format(query)
        )
        return HttpResponseRedirect('/atom')
    if request.POST.get('fulltext_search') != 'on':
        search_results = {'result': [search_results]}
    to_return = {'candidates': [
        {
            'collection': x['collection'],
            'internal_url': x['candidate']['internal-url'].split('/')[-1],
            'text': json.dumps(
                x['candidate'], sort_keys=True, indent=4, ensure_ascii=False
            )
        } for x in search_results['result']
    ]}
    return render(
        request, 'atom/visual_search.html', to_return
    )


class Candidate(object):

    def __init__(self, obj, idx):
        self.obj = obj
        self.idx = idx

    def get(self, key, default=None):
        if isinstance(self.obj, dict):
            return self.obj.get(key, default)

    def pretty(self):
        if isinstance(self.obj, dict):
            clone = {}
            for key, value in iteritems(self.obj):
                # if key in ('title', 'snippet', 'url'):
                if key not in ('filter', 'aux-data', 'text-subst'):
                    clone[key] = value
        else:
            clone = self.obj
        return json.dumps(clone, indent=4, ensure_ascii=False, sort_keys=True
                          ).replace('{\n ', '{').replace('\n}', '}')

    def getstats(self):
        result = {}
        if hasattr(self, 'statface_data'):
            sd = self.statface_data
            result['shows'] = sd.shows
            result['clicks'] = sd.clicks
            result['installs'] = sd.installs
        return result

    def displaystats(self):
        if hasattr(self, 'stats'):
            shows = int(self.stats['show'])
            if not shows:
                return '0 shows'
            ret = '{0} shows'.format(shows)
            for k, v in iteritems(self.stats):
                if k not in ['show', '__tag']:
                    try:
                        float(v)
                    except ValueError:
                        continue
                    ret += ', {0} {1} ({2:.2%})'.format(v, k, float(v) / shows)
            return 'RTMR bandit stats: ' + ret
        return ''

    def bannerid(self):
        if not isinstance(self.obj, dict):
            return None
        return self.obj.get('internal-url', '').rpartition('/')[2]

    def urlid(self):
        if not isinstance(self.obj, dict):
            return None
        u = self.obj.get('internal-url') or self.obj.get('url')
        if u and not u.startswith('banana'):
            parts = u.split('/')
            list_name_parts = parts[0].split('_')
            if len(list_name_parts) > 1 and list_name_parts[-1] in ('ru', 'tr'):
                list_name_parts.pop()
            parts[0] = '_'.join(list_name_parts)
            return '/'.join(parts)
        return u

    def toJSON(self):
        return {"idx": self.idx, "stats": self.displaystats(), "st": self.getstats(), "json": self.obj,
                "urlid": self.urlid(), "bannerid": self.bannerid()}


class EntryView(DetailView):
    model = Entry
    template_name = 'atom/entry_view.html'


def candidates_with_stats(entry):
    decoded_json = entry.candidates()
    candidates = [Candidate(c, i) for i, c in enumerate(decoded_json)]
    rtmr_key = entry.slug
    collections = set()
    for collection, sources in iteritems(get_list_sources_from_db()):
        if entry.slug in sources:
            collections.add(collection)
    if collections and rtmr_key not in collections:
        rtmr_key = collections.pop()
    try:
        j = requests.get('http://rtmr.search.yandex.net:8080/yandsearch',
                         timeout=1,
                         params=dict(view='plain', table='atom/candidate_scores',
                                     mrs=9000000,
                                     maxrecords=1, key=rtmr_key)).text.partition('\t')[2]
        stats = json.loads(j)
        # ctx['stats_json'] = json.dumps(stats, indent=2, ensure_ascii=False)
        for c in candidates:
            url = c.get('internal-url', c.get('url'))
            stat = stats.get(url, {}).get('v')
            if stat:
                c.stats = defaultdict(int,
                                      (field.split('=') for field in stat.split()))
    except Exception as e:
        return []
    candidates.sort(
        key=lambda c: int(c.stats['show']) if hasattr(c, 'stats') else 0, reverse=True)
    return candidates


def parse_field(field):
    sp = field.split('=')
    if len(sp) <= 2:
        return sp
    return ['='.join(sp[:-1]), sp[-1]]


class EntryStats(DetailView):
    model = Entry

    def get_context_data(self, **kwargs):
        ctx = super(EntryStats, self).get_context_data(**kwargs)
        obj = ctx['object']
        decoded_json = obj.candidates()
        candidates = [Candidate(c, i) for i, c in enumerate(decoded_json)]

        # attach RTMR stats
        rtmr_key = obj.slug
        collections = set()
        for collection, sources in iteritems(get_list_sources_from_db()):
            if obj.slug in sources:
                collections.add(collection)
        if collections and rtmr_key not in collections:
            rtmr_key = collections.pop()
        ctx['production_list'] = rtmr_key
        try:
            j = requests.get('http://rtmr.search.yandex.net:8080/yandsearch',
                             timeout=1,
                             params=dict(view='plain', table='atom/candidate_scores',
                                         mrs=9000000,
                                         maxrecords=1, key=rtmr_key)).text.partition('\t')[2]
            stats = json.loads(j)
            ctx['stats_json'] = json.dumps(stats, indent=2, ensure_ascii=False)
            for c in candidates:
                url = c.get('internal-url', c.get('url'))
                stat = stats.get(url, {}).get('v')
                if stat:
                    c.stats = defaultdict(
                        int, (parse_field(field) for field in stat.split())
                    )
        except Exception as e:
            pass

        # attach statface stats
        from .statface import candidate_stats
        _, _, ctx['statface_data'] = candidate_stats(obj.id)
        statface_map = {s.candidate: s for s in ctx['statface_data']}
        for c in candidates:
            b = c.bannerid()
            if b:
                s = statface_map.get(b)
                if s is not None:
                    c.statface_data = s

        candidates.sort(
            key=lambda c: int(c.stats['show']) if hasattr(c, 'stats') else 0, reverse=True)
        ctx['candidates'] = candidates
        ctx['json'] = json.dumps([c.toJSON() for c in candidates])
        return ctx


def statface_redirect(request, bannerid):
    from .statface import statface_candidate_dict
    d = statface_candidate_dict()
    url = ('https://stat.yandex-team.ru/Distribution/Others/AtomBanners/v4?'
           'scale=i&_incl_fields=cancels&_incl_fields=cancelsrate&_incl_fields=clicks'
           '&_incl_fields=clicksrate&_incl_fields=installs&_incl_fields=installsclicksrate'
           '&_incl_fields=installsrate&_incl_fields=shows'
           '&candidate={id}&_period_distance=4320'.format(
               id=d.get(bannerid)))
    return HttpResponseRedirect(url)


@require_POST
@csrf_protect
def upload_txt(request, snapshot=None):
    lists_to_revert = []
    if request.POST.get('{}_list_to_revert'.format(snapshot), ''):
        lists_to_revert = request.POST['{}_list_to_revert'.format(
            snapshot
        )].split(',')
    login = get_approved_login(request)
    if not deploy.can_freeze(login):
        messages.add_message(request, messages.ERROR,
                             'You are not authorized to perform global operations.')
        return HttpResponseRedirect('/atom')
    locks = Lock.objects.all()
    if not lists_to_revert and locks:
        messages.add_message(
            request,
            messages.ERROR,
            'Some entries are locked, unlock them and try again.'
        )
        return HttpResponseRedirect('/atom')
    elif lists_to_revert:
        bad = False
        for lock in locks:
            if lock.entry.slug in lists_to_revert:
                messages.add_message(
                    request,
                    messages.ERROR,
                    '{} is locked, unlock it and try again.'.format(
                        lists_to_revert
                    )
                )
                bad = True
        if bad:
            return HttpResponseRedirect('/atom')

    if snapshot:
        snapshot = get_object_or_404(Snapshot, pk=snapshot)
        if not os.path.isfile(snapshot.text):
            messages.add_message(
                request, messages.ERROR, 'Snapshot file does not exist.'
            )
            return HttpResponseRedirect('/atom')
        with zipfile.ZipFile(snapshot.text, 'r') as f:
            try:
                lines = f.read("snapshot.txt").decode(
                    'utf8', errors='replace'
                ).strip().split('\n')
            except:
                messages.add_message(
                    request, messages.ERROR, 'Snapshot file is corrupt.'
                )
                return HttpResponseRedirect('/atom')
        message_text = "Reverted to snapshot {}".format(snapshot.trie_name)
    else:
        try:
            f = request.FILES['candidates_txt']
        except KeyError:
            messages.add_message(
                request, messages.ERROR, 'Please provide a file to import.' + '@@'.join(request.FILES))
            return HttpResponseRedirect('/atom')
        lines = f
        message_text = "Upload successful."
    new_entries = []
    now = timezone.now()
    if not lists_to_revert:
        for i, line in enumerate(lines):
            try:
                _, slug, json_text = line.strip().split('\t')
                json_text = json.dumps(json.loads(
                    json_text), encoding='utf-8', ensure_ascii=False)
            except:
                messages.add_message(request, messages.ERROR, 'Invalid format')
                return HttpResponseRedirect('/atom')
            new_entries.append(Entry(slug=slug, json=json_text,
                                     lastchanged=now, changedby=login))
        with transaction.atomic():
            deploy.create_snapshot(
                'Automatic snapshot before checkout.', login)
            Entry.objects.all().delete()
            Entry.objects.bulk_create(new_entries)
        messages.add_message(request, messages.INFO, message_text)
        return HttpResponseRedirect('/atom')
    else:
        json_text = ''
        replacements = {}
        for i, line in enumerate(lines):
            try:
                _, slug, json_text_0 = line.strip().split('\t')
                if slug not in lists_to_revert:
                    continue
                replacements[slug] = json.loads(
                    json_text_0
                )
            except:
                messages.add_message(request, messages.ERROR, 'Invalid format')
                return HttpResponseRedirect('/atom')
        if replacements:
            for list_to_revert in replacements:
                try:
                    entry = Entry.objects.filter(slug=list_to_revert)[0]
                except:
                    messages.add_message(
                        request, messages.ERROR, 'List not found')
                    return HttpResponseRedirect('/atom')
                entry.set_candidates(replacements[list_to_revert])
                with transaction.atomic():
                    entry.reindex_urls()
                    entry.save()
                messages.add_message(
                    request,
                    messages.INFO,
                    'Succesfully reverted list {}'.format(list_to_revert)
                )
            return HttpResponseRedirect('/atom')


@require_POST
@csrf_protect
def unlock_my_entries(request):
    login = get_approved_login(request)
    if login:
        if request.POST.get('all') == 'yes':
            Lock.objects.all().delete()
        else:
            Lock.objects.filter(author=login).delete()
        messages.add_message(request, messages.INFO, 'Locks released.')
    return HttpResponseRedirect(reverse('atom:list_view'))


class Variant(text_type):
    pass


class Substitution(object):

    def __init__(self, variants={}):
        self.variants = {k: Variant(v) for k, v in iteritems(variants)}
        self.error_message = None

    def add(self, from_value, to_value=None):
        if to_value is None:
            self.variants[from_value] = Variant(from_value)
        else:
            self.variants[from_value] = Variant(to_value)

    def __getitem__(self, k):
        return self.variants.get(k, {})

    def __iter__(self):
        return iteritems(self.variants)


def extract_field_values(dst, candidate, prefix=''):
    dst = dst or defaultdict(Substitution)
    for k, v in iteritems(candidate):
        if isinstance(v, basestring):
            dst[prefix + k].add(v)
        elif isinstance(v, int) or isinstance(v, float):
            dst[prefix + k].add(str(v))
        elif isinstance(v, dict):
            extract_field_values(dst, v, prefix + k + '__')
    return dst


def simple_replace(dictionary, table, key_prefix):
    for k, v in dictionary.items():
        k_table = table.get(key_prefix + k, {})
        if isinstance(v, basestring):
            dictionary[k] = k_table.get(v, v)
        elif isinstance(v, int) or isinstance(v, float):
            replacement = k_table.get(str(v))
            if replacement:
                try:
                    if replacement.lstrip('-').isdigit():
                        dictionary[k] = int(replacement)
                    else:
                        dictionary[k] = float(replacement)
                except ValueError:
                    dictionary[k] = replacement


def replace_values_in(picked, replacements):
    for candidate in picked:
        simple_replace(candidate, replacements, '')
        aux_data = candidate.get('aux-data')
        if aux_data:
            simple_replace(aux_data, replacements, 'aux-data__')
            promolib_campaign = aux_data.get('promolib_campaign')
            if promolib_campaign:
                simple_replace(promolib_campaign, replacements,
                               'aux-data__promolib_campaign__')


def delete_candidate(url):
    candidate = find_candidate_by_id(url)
    if 'error' in candidate.keys():
        raise Exception('candidate not found')
    collection = Entry.objects.filter(slug=candidate['collection'])[0]
    candidates = json.loads(collection.json)
    candidates = [
        x for x in candidates if x['internal-url'].split('/')[-1] != url
    ]
    collection.set_candidates(candidates)
    with transaction.atomic():
        collection.reindex_urls()
        collection.save()


def replace_candidate(candidate):
    found_candidate = find_candidate_by_id(
        candidate['internal-url'].split('/')[-1]
    )
    collection = Entry.objects.filter(slug=found_candidate['collection'])[0]
    candidates = json.loads(collection.json)
    index = [
        e for e, x in enumerate(candidates)
        if x['internal-url'] == candidate['internal-url']
    ]
    candidates[index[0]] = candidate
    collection.set_candidates(candidates)
    with transaction.atomic():
        collection.reindex_urls()
        collection.save()


@require_POST
def bulk_edit_visual(request):
    action = request.POST.get('action')
    if action == 'delete':
        internal_urls = [
            c.split('candidate_')[-1]
            for c in request.POST if c.startswith('candidate')
        ]
        for url in internal_urls:
            try:
                delete_candidate(url)
            except:
                messages.add_message(
                    request, messages.ERROR, '{} not found!'.format(url)
                )
        messages.add_message(
            request, messages.INFO, 'Successfully deleted {}'.format(
                ', '.join(internal_urls)
            )
        )
        return HttpResponseRedirect('/atom')
    elif action == 'edit':
        internal_urls = [
            c.split('candidate_')[-1]
            for c in request.POST if c.startswith('candidate')
        ]
        candidates = [
            find_candidate_by_id(url)['candidate'] for url in internal_urls
        ]
        values = reduce(extract_field_values, candidates, None)
        return render(
            request, 'atom/bulk_edit_visual.html',
            {
                'values': sorted(list(values.items()), key=lambda x: x[0]),
                'internal_urls': [{'internal_url': x} for x in internal_urls]
            }
        )
    elif action == 'save':
        internal_urls = [
            c.split('candidate_')[-1]
            for c in request.POST if c.startswith('candidate')
        ]
        candidates = [find_candidate_by_id(url) for url in internal_urls]
        picked = [x['candidate'] for x in candidates]
        replacements = defaultdict(dict)
        all_replacements = defaultdict(Substitution)
        for k, vs in request.POST.iterlists():
            if k.endswith('__replacefrom'):
                varname = k.rpartition('__')[0]
                replaceto = request.POST.getlist(varname + '__replaceto')
                for f, t in zip(vs, replaceto):
                    if f != t:
                        replacements[varname][f] = t
                    all_replacements[varname].add(f, t)

        has_errors = False

        filters = all_replacements.get('filter')
        if filters:
            for k, v in iteritems(filters.variants):
                vf, ps = valid_filter(v)
                if not vf:
                    v.error_message = "Invalid filter, problem location: %s." % (
                        ps)
                    has_errors = True

        if has_errors:
            return render(
                request, 'atom/bulk_edit_visual.html',
                {
                    'values': dict(all_replacements),
                    'internal_urls': [
                        {'internal_url': x} for x in internal_urls
                    ]
                }
            )

        replace_values_in(picked, replacements)
        for p in picked:
            replace_candidate(p)

        messages.add_message(
            request, messages.INFO, 'Succesfully replaced values!'
        )

    return HttpResponseRedirect('/atom')


@require_POST
def bulk_edit(request, pk):
    lock = None
    json.dump(dict(request.POST), open('request_bulk_edit.json', 'w'))
    collection = get_object_or_404(Entry, pk=pk)
    login = get_approved_login(request)
    if not login:
        return HttpResponseRedirect('atom:list_view')

    lock, created = Lock.objects.get_or_create(entry=collection,
                                               defaults={'author': login, 'created': timezone.now()})
    if lock.author != login:
        messages.add_message(request, messages.ERROR,
                             'This entry is locked by ' + lock.author)
        return HttpResponseRedirect(reverse('atom:list_view'))

    indices = [int(k[4:]) for k, v in iteritems(request.POST)
               if k.startswith('edit') and v == 'on']
    candidates = collection.candidates()
    picked = [candidates[i] for i in indices]

    action = request.POST.get('action')
    if action == 'delete':
        candidates = [c for (i, c) in enumerate(
            candidates) if i not in indices]
        collection.set_candidates(candidates)
        with transaction.atomic():
            collection.reindex_urls()
            collection.save()
    elif action == 'edit':
        values = reduce(extract_field_values, picked, None)
        return render(
            request, 'atom/bulk_edit.html',
            {
                'values': sorted(list(values.items()), key=lambda x: x[0]),
                'slug': collection.slug,
                'indices': indices,
                'source': collection,
                'entries': sorted(
                    [{'slug': x.slug} for x in Entry.objects.all()]
                )
            }
        )
    elif action == 'save':
        replacements = defaultdict(dict)
        all_replacements = defaultdict(Substitution)
        for k, vs in request.POST.iterlists():
            if k.endswith('__replacefrom'):
                varname = k.rpartition('__')[0]
                replaceto = request.POST.getlist(varname + '__replaceto')
                for f, t in zip(vs, replaceto):
                    if f != t:
                        replacements[varname][f] = t
                    all_replacements[varname].add(f, t)

        has_errors = False

        to_bannerids = [url.partition('/')[2]
                        for url in replacements.get('internal-url', {}).values()]
        dup_error_message = get_dups(pk, to_bannerids)
        if dup_error_message:
            all_replacements['internal-url'].error_message = dup_error_message
            has_errors = True

        filters = all_replacements.get('filter')
        if filters:
            for k, v in iteritems(filters.variants):
                vf, ps = valid_filter(v)
                if not vf:
                    v.error_message = "Invalid filter, problem location: %s." % (
                        ps)
                    has_errors = True

        if has_errors:
            messages.add_message(
                request, messages.ERROR,
                'Invalid filter or duplicate bannerids'
            )
            return HttpResponseRedirect(reverse('atom:list_view'))

        replace_values_in(picked, replacements)
        destination = request.POST.get('destination', '')
        if destination:
            dest_entry = Entry.objects.filter(slug=destination)[0]
            dest_candidates = dest_entry.candidates()
            bannerids = get_n_bannerids(len(picked))
            for i, c in enumerate(picked):
                c['internal-url'] = (
                    c['internal-url'].split('/')[0] + '/' + bannerids[i]
                )
            dest_candidates.extend(picked)
            dest_entry.set_candidates(dest_candidates)
            with transaction.atomic():
                dest_entry.reindex_urls()
                dest_entry.save()
        else:
            for e, i in enumerate(indices):
                candidates[i] = picked[e]
            collection.set_candidates(candidates)
            with transaction.atomic():
                collection.reindex_urls()
                collection.save()
    elif action == 'confirm':
        destination = request.POST.get('destination', '')
        if destination:
            texts = []
            for k, v in iteritems(request.POST):
                if k.startswith('approved_'):
                    texts.append((0, v))
            return add_texts_to_entries(login, texts, [destination])
        for k, v in iteritems(request.POST):
            if k.startswith('approved_'):
                idx = int(k.partition('_')[2])
                candidates[idx] = json.loads(v)
        collection.set_candidates(candidates)
        with transaction.atomic():
            collection.reindex_urls()
            collection.save()

    EditLog.add(login, collection.slug)
    if lock:
        lock.delete()
    return HttpResponseRedirect(reverse('atom:entry_stats', args=[pk]))


def save_entry_form(form, login):
    instance = form.save(commit=False)
    instance.changedby = login
    with transaction.atomic():
        instance.save()
        instance.reindex_urls()
        form.save_m2m()
    EditLog.add(login, instance.slug)


def entry_edit(request, entry_id=None):
    login = get_approved_login(request)
    if not login:
        return HttpResponseRedirect(reverse('atom:list_view'))
    lock = None

    if entry_id:
        e = get_object_or_404(Entry, pk=entry_id)
        e.changedby = login
        try:
            lock, created = Lock.objects.get_or_create(
                entry=e, defaults={'author': login})
            if lock.author != login:
                messages.add_message(
                    request, messages.ERROR, 'This entry is locked by ' + lock.author)
                return HttpResponseRedirect(reverse('atom:list_view'))
        except Exception as e:
            messages.add_message(request, messages.ERROR,
                                 'Failed to lock entry.')
            return HttpResponseRedirect(reverse('atom:list_view'))
    else:
        e = None
    if request.method == 'POST':
        success = False
        if 'delete' in request.POST:
            e.delete()
            success = True
        elif 'unlock' in request.POST:
            success = True
        else:
            form = EntryForm(request.POST, instance=e)
            if form.is_valid():
                save_entry_form(form, login)
                success = True
        if success:
            if lock:
                lock.delete()
            return HttpResponseRedirect(reverse('atom:list_view'))
    else:
        form = EntryForm(instance=e)
    return render(request, 'atom/edit.html',
                  dict(form=form, title=(form.instance.slug or "New list")))


def const_add(request, entry_id=None):
    login = get_approved_login(request)
    if not login:
        return HttpResponseRedirect(reverse('atom:list_view'))
    lock = None

    if entry_id:
        e = get_object_or_404(Entry, pk=entry_id)
        e.changedby = login
        try:
            lock, created = Lock.objects.get_or_create(
                entry=e, defaults={'author': login})
            if lock.author != login:
                messages.add_message(
                    request, messages.ERROR, 'This entry is locked by ' + lock.author)
                return HttpResponseRedirect(reverse('atom:list_view'))
        except Exception as e:
            messages.add_message(request, messages.ERROR,
                                 'Failed to lock entry.')
            return HttpResponseRedirect(reverse('atom:list_view'))
    else:
        e = None
    if request.method == 'POST':
        success = False
        if 'cancel' in request.POST:
            success = True
        elif 'const_add' in request.POST:
            form = EntryForm(request.POST, instance=e)
            try:
                obj = json.loads(form.cleaned_data.get('json'))
            except:
                messages.add_message(
                    request, messages.ERROR, traceback.format_exc())
                return HttpResponseRedirect(reverse('atom:list_view'))
            obj1 = e.candidates()
            if isinstance(obj, dict):
                obj1.append(obj)
            elif isinstance(obj, list):
                obj1.extend(obj)
            e.set_candidates(obj1)
            with transaction.atomic():
                e.reindex_urls()
                e.save()
            deployment = deploy.create_deployment(login, 'const_add')
            messages.add_message(request, messages.INFO, 'Deployment started.')
            two_step_rollout.delay(None, login, deployment.pk)
        if success:
            if lock:
                lock.delete()
            return HttpResponseRedirect(reverse('atom:list_view'))
    else:
        form = EntryForm(instance=e)
    return render(request, 'atom/const_add.html',
                  dict(form={'slug': e.slug, 'json': ''}, title=(form.instance.slug or "New list")))


@require_POST
@csrf_protect
def revert_to(request, snapshot):
    snapshot = get_object_or_404(Snapshot, pk=snapshot)


def dump(request, snapshot=None):
    if snapshot:
        snapshot = get_object_or_404(Snapshot, pk=snapshot)

    response = HttpResponse(content_type='text/plain; charset=UTF-8')
    response['Content-Disposition'] = 'attachment; filename="candidates{}.txt"'.format(
        '-' + snapshot.trie_name if snapshot else '')
    if snapshot:
        response.write(snapshot.text.encode('utf-8'))
    else:
        deploy.dump_entries_to(response)
    return response


def dump_zip(request, snapshot):
    snapshot = get_object_or_404(Snapshot, pk=snapshot)

    response = HttpResponse(content_type='application/x-zip-compressed')
    response['Content-Disposition'] = 'attachment; filename="candidates-{}.zip"'.format(
        snapshot.trie_name)
    response.write(snapshot.zipped())
    return response


@require_POST
def force_deploy(request, depl):
    d = get_object_or_404(Deployment, pk=depl)
    d.is_rollback = True
    d.save(update_fields=['is_rollback'])
    return HttpResponseRedirect(reverse('atom:logs'))


@require_POST
def fail_deploy(request, depl): #TODO: сделать, чтоб таска в celery тоже прибивалась этой функцией
    Deployment.objects.filter(pk=depl).update(state='failure')
    return HttpResponseRedirect(reverse('atom:logs'))


@require_POST
@csrf_protect
def rollout(request):
    login = get_approved_login(request)
    if not login:
        return HttpResponseRedirect(reverse('atom:list_view'))
    freeze = GlobalVar.get('freeze')
    if freeze and freeze.get_value():
        messages.add_message(request, messages.ERROR, 'Deployments frozen.')
        return HttpResponseRedirect(reverse('atom:list_view'))

    comment = request.POST.get('comment', '')
    deployment = deploy.create_deployment(login, comment)
    messages.add_message(request, messages.INFO, 'Deployment started.')
    two_step_rollout.delay(None, login, deployment.pk)

    return HttpResponseRedirect(reverse('atom:list_view'))


class EditorCreate(CreateView):
    model = Editor
    fields = ['login', 'allowed_slugs']
    success_url = reverse_lazy('atom:editor_list')


class EditorDelete(DeleteView):
    model = Editor
    success_url = reverse_lazy('atom:editor_list')


class EditorList(ListView):
    model = Editor
    createview = EditorCreate()
    deleteview = EditorDelete()

    def get_context_data(self, **kwargs):
        context = super(EditorList, self).get_context_data(**kwargs)
        context['add_form'] = self.createview.get_form_class()()
        return context


class SnapshotList(ListView):
    model = Snapshot
    paginate_by = 20

    def get_queryset(self):
        return self.model.objects.defer('text')

    def get_context_data(self, **kwargs):
        context = super(SnapshotList, self).get_context_data(**kwargs)
        context['can_freeze'] = deploy.can_freeze(
            get_approved_login(self.request))
        return context


class LogList(ListView):
    model = Deployment
    ordering = '-created'
    paginate_by = 10
    template_name = 'atom/logs.html'

    def get_queryset(self):
        return super(LogList, self).get_queryset().prefetch_related('deploylog_set')


class TemplateList(ListView):
    model = BannerTemplate


class TemplateCreate(CreateView):
    model = BannerTemplate
    form_class = BannerTemplateForm
    success_url = reverse_lazy('atom:template_list')


class TemplateUpdate(UpdateView):
    model = BannerTemplate
    form_class = BannerTemplateForm
    success_url = reverse_lazy('atom:template_list')


class TemplateDelete(DeleteView):
    model = BannerTemplate
    success_url = reverse_lazy('atom:template_list')


class AutoQuotaView:
    model = AutoQuota
    form_class = AutoQuotaForm
    slug_field = 'production_list'
    success_url = reverse_lazy('atom:quota')


class AutoQuotaCreate(AutoQuotaView, CreateView):
    pass


class AutoQuotaUpdate(AutoQuotaView, UpdateView):
    pass


class AutoQuotaDelete(AutoQuotaView, DeleteView):
    pass


def choose_list(request, slug):
    return render(request, 'atom/choose_list.html',
                  dict(lists=Entry.objects.all(), template=slug))


def pick_candidates(request, slug, list_id):
    t = get_object_or_404(BannerTemplate, slug=slug)
    lst = get_object_or_404(Entry, pk=list_id)
    decoded_json = lst.candidates()
    candidates = map(Candidate, decoded_json, range(len(decoded_json)))
    return render(request, 'atom/pick_candidates.html',
                  dict(template=t, lst=lst, candidates=candidates))


def substitutions(request, slug, list_id):
    t = get_object_or_404(BannerTemplate, slug=slug)
    lst = get_object_or_404(Entry, pk=list_id)
    if request.POST.get('action') != 'edit':
        var_lists = t.get_variables(lst.candidates())
    else:
        var_lists = t.get_variables([c for (i, c) in enumerate(lst.candidates())
                                     if request.POST.get('edit%s' % i) == 'on'])
    return render(request, 'atom/substitutions.html',
                  dict(template=t, var_lists=var_lists, lst=list_id))


def var_values(var, text):
    return izip(repeat(var),
                filter(None, [line.strip() for line in text.split('\n')])
                or [''])


def combinations(request, slug):
    t = get_object_or_404(BannerTemplate, slug=slug)
    var_lists = [var_values(key[4:], text) for key, text in iteritems(request.POST)
                 if key.startswith('var_')]
    final_texts = enumerate(chain(*[t.replace(vl)
                                    for vl in product(*var_lists)]))
    return render(request, 'atom/approve.html',
                  dict(texts=final_texts, lists=Entry.objects.all()))


def add_texts_to_entry(e, login, add_texts):
    j = e.candidates()

    bannerids = get_n_bannerids(len(add_texts))
    ready_candidates = [
        json.loads(
            text[1].replace('{{banner_id}}', bannerids[i]))
        for i, text in enumerate(add_texts)]

    for i, candidate in enumerate(ready_candidates):
        if isinstance(candidate, dict):
            internal_url = candidate.get('internal-url', '')
            if bannerids[i] not in internal_url:
                candidate['internal-url'] = '/'.join((
                    internal_url.rpartition('/')[0],
                    bannerids[i]))

    j.extend(ready_candidates)
    e.set_candidates(j)
    with transaction.atomic():
        e.reindex_urls()
        e.author = login
        e.save()
    EditLog.add(login, e.slug)


def add_creatives(request):
    list_ids = request.POST.getlist('destination_list')
    if not list_ids:
        return HttpResponseRedirect(reverse('atom:template_list'))

    login = get_approved_login(request)
    if not login:
        return HttpResponseRedirect(reverse('atom:list_view'))

    add_texts = []
    for k, v in iteritems(request.POST):
        if not (k.startswith('use') and v == 'on'):
            continue
        text_id = k[3:]
        add_texts.append([int(text_id), request.POST['text' + text_id]])
    add_texts.sort()
    return add_texts_to_entries(login, add_texts, list_ids)


def add_texts_to_entries(login, add_texts, list_ids):
    for list_id in list_ids:
        e = get_object_or_404(Entry, pk=list_id)
        add_texts_to_entry(e, login, add_texts)
    if len(list_ids) == 1:
        return HttpResponseRedirect(reverse('atom:entry_view', args=[list_ids[0]]))
    return HttpResponseRedirect(reverse('atom:list_view'))


def heartbeat(request):
    return HttpResponse('ok')


@python_2_unicode_compatible
class Query(object):

    def __init__(self, q):
        self.words = q.strip().split()

    def __str__(self):
        return ' '.join(self.words)


def minus(request):
    if request.method == 'GET':
        return render(request, 'atom/minus_form.html')
    if request.method == 'POST':
        queries = [Query(q) for q in request.POST[
            'queries'].split('\n') if q.strip()]
        word_dict = defaultdict(set)
        for q in queries:
            for word in q.words:
                if len(word) > 2:
                    word_dict[word].add(q)
        for q in queries:
            other_sets = [word_dict[word] for word in q.words]
            others = reduce(operator.and_, other_sets)
            others.difference_update([q])
            minus_words = set()
            for o in others:
                minus_words.update(w for w in o.words if not w.startswith('-'))
            minus_words.difference_update(q.words)
            q.words.extend('-' + w for w in minus_words)
        return render(request, 'atom/minus_result.html', {'queries': '\n'.join(unicode(q) for q in queries)})


def formatted_json_lines(j):
    return json.dumps(json.loads(j),
                      ensure_ascii=False, indent=2, sort_keys=True).split('\n')


def md5(x):
    return hashlib.md5(x).hexdigest()


def fast_diff(x1, x2):
    with codecs.open('x1', 'w', 'utf8') as f:
        f.write('\n'.join(x1))
    with codecs.open('x2', 'w', 'utf8') as f:
        f.write('\n'.join(x2))
    os.system('diff x1 x2 > x3')
    with codecs.open('x3', 'r', 'utf8') as f:
        out = f.read()
    os.remove('x1')
    os.remove('x2')
    os.remove('x3')
    return out


def get_snapshots_by_date(date_start, date_end):
    try:
        s1 = Snapshot.objects.filter(
            created__lte=date_start
        ).order_by('-created')[0]
        s2 = Snapshot.objects.filter(
            created__lte=date_end
        ).order_by('-created')[0]
        return s1, s2
    except IndexError:
        return HttpResponseRedirect('/atom')


def diff(request, s1=None, s2=None):
    s1 = s1 or request.GET.get('oldid')
    s2 = s2 or request.GET.get('diff')
    date_start = request.GET.get('date_start')
    date_end = request.GET.get('date_end')
    if (
        date_start == 'start date' or not date_start or
        date_end == 'end date' or not date_end
    ):
        o1 = get_object_or_404(Snapshot, pk=s1)
        o2 = get_object_or_404(Snapshot, pk=s2)
    else:
        try:
            date_start = datetime.datetime.strptime(
                date_start, '%Y-%m-%d %H:%M'
            )
            date_end = datetime.datetime.strptime(
                date_end, '%Y-%m-%d %H:%M'
            )
        except:
            messages.add_message(
                request, messages.ERROR,
                'one of dates {}, {} is unparsable.'.format(
                    date_start, date_end
                )
            )
            return HttpResponseRedirect('/atom')
        o1, o2 = get_snapshots_by_date(date_start, date_end)
    s1 = o1.as_dict()
    s2 = o2.as_dict()
    keys1 = {k for k in s1 if s1[k] != '[]'}
    keys2 = {k for k in s2 if s2[k] != '[]'}
    deleted = ', '.join(sorted(keys1 - keys2))
    added = ', '.join(sorted(keys2 - keys1))
    keys_to_diff = sorted({
        k for k in (keys1 & keys2) if s1.get(k, u'[]') != s2.get(k, u'[]')
    })
    differ = difflib.HtmlDiff(wrapcolumn=int(request.GET.get('wrap', 80)))
    v1s = (formatted_json_lines(s1.get(k, '[]')) for k in keys_to_diff)
    v2s = (formatted_json_lines(s2.get(k, '[]')) for k in keys_to_diff)

    def make_table(*args, **kwargs):
        if max(len(a) for a in args) > 1000:
            # return u'<pre>{}</pre>'.format(diffs.unidiff(*args).decode('utf-8'))
            return u'<pre>{}</pre>'.format(fast_diff(*args))
        return differ.make_table(*args, **kwargs)
    hunks = (
        {
            'key': k,
            'diff': make_table(v1, v2, context=True)
        }
        for (k, v1, v2) in izip(keys_to_diff, v1s, v2s)
    )
    return render(
        request,
        'atom/diff.html',
        {
            'hunks': hunks,
            's1': o1,
            's2': o2,
            'deleted': deleted,
            'added': added
        }
    )


def disguise(request):
    if request.method != 'POST':
        return render(request, 'atom/disguise.html', {})
    request.session['yandex_login'] = request.POST['yandex_login']
    return HttpResponseRedirect(reverse('atom:list_view'))


@require_POST
@csrf_protect
def freeze(request, setting):
    login = get_approved_login(request)
    if deploy.can_freeze(login):
        deploy.set_freeze(setting, login)
    else:
        messages.add_message(request, messages.ERROR, 'Permission denied.')
    return HttpResponseRedirect(reverse('atom:list_view'))


def top(request):
    from .statface import join_authors
    aggregate = join_authors()
    items = sorted([d for d in aggregate.data if d.author],
                   key=lambda i: i.ctr(), reverse=True)
    return render(request, 'atom/leaderboard.html', {
        'since': aggregate.since, 'until': aggregate.until, 'items': items})


def top_candidates(request):
    from .statface import candidate_top
    product_filter = request.GET.get('product', '')
    since, until, top = candidate_top(product=product_filter)
    collection_filter = request.GET.get('collection', '')
    list_sources = get_list_sources_from_db()
    products = CandidateUrl.objects.values_list('product', flat=True).filter(
        product__isnull=False).distinct()
    if collection_filter:
        list_slugs = list_sources.get(collection_filter, [])
        list_ids = Entry.objects.values_list(
            'id', flat=True).filter(slug__in=list_slugs)
        products = products.filter(list_id__in=list_ids)
        top = [t for t in top if t.info and t.list_id in list_ids]
    else:
        top = top[:200]
    urls = set(t.candidate for t in top)
    url_records = {c.url: c for c in CandidateUrl.objects.filter(url__in=urls)}
    for t in top:
        t.info = url_records.get(t.candidate)
    return render(request, 'atom/c_leaderboard.html', {
        'since': since, 'until': until, 'top': top, 'collections': sorted(list_sources),
        'products': products, 'product_filter': product_filter,
        'current_filter': collection_filter})


def quota(request):
    return render(request, 'atom/quota.html',
                  {'objects': AutoQuota.objects.order_by('production_list')})


QUOTA_DIR = '/home/atom_admin/app/arcadia_quota'
SVN_SSH = 'SVN_SSH="ssh -i atom-releases -l robot-atom-releases"'


def get_quota_value():
    cwd = os.getcwd()
    os.chdir(QUOTA_DIR)
    os.system('{} svn revert quota.txt'.format(SVN_SSH))
    os.system('{} svn up .'.format(SVN_SSH))
    contents = ''
    with codecs.open('quota.txt', 'r', 'utf8') as f:
        contents = f.read()
    os.chdir(cwd)
    return contents


def set_quota_value(value, login='unknown'):
    cwd = os.getcwd()
    os.chdir(QUOTA_DIR)
    with codecs.open('quota.txt', 'w', 'utf8') as f:
        f.write(quota_preprocess(value))
    os.system('{} svn ci -m "{}@ edited quota online"'.format(
        SVN_SSH, login
    ))
    os.chdir(cwd)


def quota_preprocess(value):
    value = value.replace('\r\n', '\n').strip()
    lines = value.split('\n')
    for line in lines:
        if not line:
            continue
        while line[-1] in {' ', '\t'}:
            line = line[:-1]
    value = '\n'.join(lines) + '\n'
    return value


def quota_edit(request):
    login = get_approved_login(request)
    if not login:
        return HttpResponseRedirect('/atom')
    if request.method == 'POST':
        action = request.POST.get('action', '')
        if action == 'save':
            value = request.POST.get('value', '')
            if value:
                set_quota_value(value, login=login)
    return render(
        request, 'atom/quota_edit.html',
        {'quota': {
            'value': get_quota_value(),
            'message': ''
        }}
    )


def find_candidate_by_id(url):
    for entry in Entry.objects.all():
        if url not in entry.json:
            continue
        coll = json.loads(entry.json)
        for obj in coll:
            try:
                if obj['internal-url'].split('/')[-1] == url:
                    return {
                        'collection': entry.slug,
                        'candidate': obj
                    }
            except:
                pass
    return {'error': 'candidate not found :('}


def find_candidate_by_fulltext(fulltext):
    result = []
    for entry in Entry.objects.all():
        coll = json.loads(entry.json)
        for obj in coll:
            try:
                if (
                    fulltext in json.dumps(obj, ensure_ascii=False) or
                    fulltext.lower() in json.dumps(
                        obj, ensure_ascii=False
                    ).lower()
                ):
                    result.append({
                        'collection': entry.slug,
                        'candidate': obj
                    })
                    if len(result) >= 100:
                        return {'result': result}
            except:
                pass
    if result:
        return {'result': result}
    return {'error': 'candidate not found :('}


def get_all_internal_urls():
    result = {}
    for entry in Entry.objects.all():
        if entry.slug == 'all_keys':
            continue
        coll = json.loads(entry.json)
        result[entry.slug] = [
            cand['internal-url'].split('/')[-1] for cand in coll
        ]
    return result


@csrf_exempt
def api(request, endpoint):
    ep = endpoint.rstrip('/').partition('/')[::2]
    login = get_approved_login(request)

    if ep[0] == 'token':
        if login:
            t, created = AuthToken.objects.get_or_create(login=login)
            t = t.key
        else:
            t = 'You are not logged in @yandex-team.ru.'
        return HttpResponse(t)
    if ep[0] == 'collections':
        if ep[1]:
            slug = ep[1]
            if request.method in ('POST', 'PUT', 'DELETE'):
                if not login:
                    return HttpResponseNotAllowed('not logged in')
                instance = None
                if request.method in ('PUT', 'DELETE'):
                    instance = get_object_or_404(Entry, slug=slug)
                if request.method == 'DELETE':
                    instance.delete()
                    return HttpResponse()
                form = EntryForm(
                    {'slug': slug, 'json': request.body}, instance=instance)
                if form.is_valid():
                    save_entry_form(form, login)
                    return HttpResponse()
                else:
                    j = JsonResponse({'errors': [form.errors]})
                    j.status_code = 400
                    return j
            else:  # GET
                e = get_object_or_404(Entry, slug=slug)
                return JsonResponse(e.candidates(), safe=False)

        else:  # no ep[1]
            my_slugs = get_recently_edited(login)
            entries = dict((e.slug, e)
                           for e in Entry.objects.all().defer('json'))

            all_keys = entries.get('all_keys')
            all_keys = set(all_keys.candidates() if all_keys else [])

            response = dict((e.slug,
                             {'id': e.slug, 'pk': e.pk, 'changedby': e.changedby,
                              'lastchanged': '',
                              'enabled': e.slug in all_keys,
                              'my': e.slug in my_slugs})
                            for e in entries.values())
            for l in Lock.objects.all():
                response[l.entry.slug]['is_locked'] = True
                response[l.entry.slug]['lock'] = {
                    'author': l.author, 'created': '2015'}

            return HttpResponse(json.dumps(response.values()))
        return HttpResponse(Entry.objects.get(slug=ep[1]).json)
    elif ep[0] == 'lists':
        if ep[1]:
            slug = ep[1]
            e = get_object_or_404(Entry, slug=slug)
            candidates = candidates_with_stats(e)
            return JsonResponse([c.toJSON() for c in candidates], safe=False)
    elif ep[0] == 'version' and ep[1] == 'production':
        try:
            d = Deployment.objects.filter(state='success').latest('created')
            if d.task_id:
                return JsonResponse({'task_id': d.task_id})
            if d.snapshot.task_id:
                return JsonResponse({'task_id': d.snapshot.task_id})
            d = Deployment.objects.filter(
                snapshot__task_id__isnull=False).latest()
            return JsonResponse({'task_id': d.snapshot.task_id})
        except Deployment.DoesNotExist:
            return JsonResponse({})
    elif ep[0] == 'products':
        products = defaultdict(set)
        for c in CandidateUrl.objects.values('url', 'product'):
            products[c['product']].add(c['url'])
        return HttpResponse('\n'.join('{product}\t{bannerids}'.format(
            product=product, bannerids='\t'.join(ps))
            for product, ps in products.iteritems()
            if product is not None), 'text/plain')
    elif ep[0] == 'logs':
        return JsonResponse(
            {
                'deployments': [
                    {
                        'task_id': x.task_id,
                        'author': x.author,
                        'created': int(x.created.strftime('%s'))
                    }
                    for x in Deployment.objects.all().order_by('-created')[:15]
                ]
            }
        )
    elif ep[0] == 'find':
        try:
            return JsonResponse(
                find_candidate_by_id(ep[1].replace(' ', '')),
                json_dumps_params={
                    'indent': 4,
                    'ensure_ascii': False
                }
            )
        except IndexError:
            pass
    elif ep[0] == 'find_fulltext':
        try:
            return JsonResponse(
                find_candidate_by_fulltext(ep[1]),
                json_dumps_params={
                    'indent': 4,
                    'ensure_ascii': False
                }
            )
        except IndexError:
            pass
    elif ep[0] == 'all_internal_urls':
        return JsonResponse(get_all_internal_urls())
    elif ep[0] == 'nbannerids':
        try:
            n = int(ep[1])
        except:
            return JsonResponse({'error': 'integer not specified'})
        return JsonResponse({'bannerids': get_n_bannerids(n)})
    elif ep[0] == 'deploy':
        login = get_approved_login(request)
        if not login:
            return JsonResponse(
                {
                    'error': 'login not approved'
                }
            )
        try:
            message = request.POST['message']
        except (AttributeError, KeyError):
            return JsonResponse(
                {
                    'error': 'type of request should be POST '
                    'and message should be passed.'
                }
            )
        deployment = deploy.create_deployment(login, message)
        two_step_rollout.delay(None, login + '_api', deployment.pk)
        return JsonResponse(
            {
                'success': 'Deploy has started.'
            }
        )

    return HttpResponse(endpoint)
