# coding=utf-8
from __future__ import division
from datetime import date, timedelta
import api_connector as facebook_api_connector
from facebookads.objects import AdCampaign, AdSet, TargetingSpecsField
from ..metrika import conversions
import logging, os
import targeting_split

logger = logging.getLogger('ru.yandex.facebook.campaign_management')

__author__ = 'aalogachev'


def get_action_type_value(actions, action_type):
    for action in actions:
        if action.get('action_type') == action_type:
            return action.get('value')
    return 0


def get_cpc_and_money_spent(adset, start_date, end_date):
    params = {
        'time_range': {
            'since': start_date.strftime('%Y-%m-%d'),
            'until': end_date.strftime('%Y-%m-%d'),
        },
        'fields': ['campaign_group_name', 'relevance_score', 'impressions', 'website_clicks', 'actions', 'spend', 'cpm',
                   'cpc', 'ctr', 'cost_per_action_type', 'website_ctr'],
    }
    report = adset.get_insights(params=params)

    # initial return values
    res = 0
    spent = 0

    for item in report:
        cost_actions = item.get('cost_per_action_type', None)
        if cost_actions is not None:
            res = get_action_type_value(cost_actions, 'link_click')
        spent = item.get('spend')
    return res, spent


def update_campaign_cpc_for_day(campaign_id, dry_run):
    update_logger = logging.getLogger('ru.yandex.facebook.campaign_management.update_logger')
    formatter = logging.Formatter('%(asctime)s : %(message)s')
    fileHandler = logging.FileHandler(os.path.expanduser('~/facebook_update.log'), mode='w')
    fileHandler.setFormatter(formatter)
    update_logger.addHandler(fileHandler)

    DESIRED_CPI = 1
    MAX_BID_RAISE_RATIO = 1.5
    MAX_CONVERSION_RATE = 0.5
    DEFAULT_AMNISTION = 2  # using default big value for amnistion to get statistics for it and for conversions
    # budget constants in cents
    MAX_DAILY_BUDGET = 200 * 100
    DELTA_BUDGET_UPDATE = 20 * 100
    BUDGET_EPSILON = 1 * 100

    fb_connector = facebook_api_connector.ApiConnector()

    campaign = AdCampaign(campaign_id)
    adsets = campaign.get_ad_sets()

    account = fb_connector.get_account()
    api_batch = account.get_api_assured().new_batch()

    def callback_failure(response):
        logger.error(response)
        raise response.error()

    for adset in adsets:
        #if adset['id'] not in ('6026897717408', '6026898649608', '6026898869808'): continue
        # targeting = adset.get(AdSet.Field.targeting)
        # age_min = targeting.get(TargetingSpecsField.age_min)
        # age_max = targeting.get(TargetingSpecsField.age_max)
        # genders = targeting.get(TargetingSpecsField.genders)
        # age = u'{0}-{1}'.format(age_min, age_max)
        # gender = None if len(genders) > 1 else genders[0]
        logger.info(u'{0} - {1}'.format(adset['id'], adset['name']))

        stat_date = date.today() - timedelta(days=1)
        # getting more statistics to make accurate perdictions
        v_cvr = conversions.get_visits_and_conversion_rates_for(adset['id'], conversions.SWITCH_COUNTER_ID,
                                                                conversions.SWITCH_INSTALL_GOAL_ID,
                                                                stat_date, stat_date)
        v_cvr_stat = v_cvr
        delta_period_days = 1
        # getting more statistics for previous periods
        while (v_cvr_stat[0] < 50 and delta_period_days < 7):
            v_cvr_stat = conversions.get_visits_and_conversion_rates_for(adset['id'], conversions.SWITCH_COUNTER_ID,
                                                                         conversions.SWITCH_INSTALL_GOAL_ID,
                                                                         stat_date - timedelta(
                                                                             days=delta_period_days - 1), stat_date)
            delta_period_days += 1
        #logger.info(u'Conversion estimate period in days was = {0}'.format(delta_period_days))
        logger.info(u'visits = {0}; conversion_rate = {1} in stat_date=[{2}]'.format(v_cvr[0], v_cvr[1], stat_date))
        logger.info(u'visits = {0}; conversion_rate = {1} in stat_period=[{2}-{3}]'.format(v_cvr_stat[0], v_cvr_stat[1],
                    stat_date, stat_date - timedelta(days=delta_period_days - 1)))
        #calculating cpc estimation
        cpc_est, cpc_delta_est = conversions.calculate_cpc_estimate_and_delta(v_cvr_stat, DESIRED_CPI)
        logger.info(u'CPC_est = {1}\u00B1{2}'.format(0, cpc_est, cpc_delta_est))
        cpc_current_fb, money_spent = get_cpc_and_money_spent(adset, stat_date, stat_date)

        if v_cvr[0] != 0:
            cpc_current = money_spent / v_cvr[0]
        elif money_spent > 0:
            # FIXME think about cpc_current calculation in this case
            cpc_current = DESIRED_CPI

        logger.info(u'CPC_current_fb = {0} and CPC_current_metrics = {1}'.format(cpc_current_fb, cpc_current))

        cpc_to_set = cpc_est + cpc_delta_est
        cpc_bid = adset.get(AdSet.Field.bid_amount) / 100
        amnistion = DEFAULT_AMNISTION
        if cpc_current != 0:
            amnistion = cpc_bid / cpc_current
        cpc_to_set = cpc_to_set * amnistion

        # limiting max cpc, limiting max increase
        cpc_to_set = min(cpc_to_set, DESIRED_CPI * MAX_CONVERSION_RATE, cpc_bid * MAX_BID_RAISE_RATIO)

        logger.info(u'Bid was {0}'.format(cpc_bid))
        log_and_perform_update(
            adset, update_logger, dry_run, {
                AdSet.Field.bid_amount: int(cpc_to_set * 100),
            })
        logger.info(u'Bid set {0}'.format(cpc_to_set))

        # updating budget
        current_daily_budget = float(adset.get(AdSet.Field.daily_budget)) / 100
        logger.info(u'Daily budget was {0}'.format(current_daily_budget))
        logger.info(u'Money spent {0}'.format(money_spent))
        if (money_spent + BUDGET_EPSILON > current_daily_budget) and (
                        cpc_est + cpc_delta_est > cpc_current) and MAX_DAILY_BUDGET < current_daily_budget:
            # adding more daily budget (we spend it all and have good CPA price), so increasing volumes
            budget_to_set = current_daily_budget + DELTA_BUDGET_UPDATE
            budget_to_set = min(budget_to_set, MAX_DAILY_BUDGET)
            log_and_perform_update(adset, update_logger, dry_run, {
                AdSet.Field.daily_budget: budget_to_set
            })
            logger.info(u'Daily budget set {0}'.format(budget_to_set))
        else:
            logger.info(u'Not updating budget')
        adset.remote_update(batch=api_batch)

    # executing batch update
    targeting_split.bacth_execute_safe(api_batch)


def log_and_perform_update(adobject, update_logger, dry_run, map_field_values):
    # if dry run than do nothing
    if dry_run: return
    for field, value in map_field_values.iteritems():
        update_logger.info(u'type=update\tobj={0}\tid={1}\tfield={2}\tprev_field_value={3}\tcurrent_field_value={4}'.
                           format(adobject.__class__.__name__, adobject.get('id'), field, adobject.get(field), value))
    adobject.update(map_field_values)
    return adobject


if __name__ == '__main__':
    logging.basicConfig(format='%(asctime)s : %(levelname)s : %(message)s', level=logging.INFO)
    update_campaign_cpc_for_day(6026896277808, dry_run=False)

