# coding=utf-8
from __future__ import division
from facebookads import FacebookAdsApi, FacebookSession
from facebookads.objects import TargetingSpecsField, TargetingDescription, TargetingSearch, AdCampaign, AdSet, AdGroup, \
    AdCreative, AdImage, AdUser, AdAccount, AdLabel
from facebookads.exceptions import FacebookRequestError
from dimensions import Dimension, TargetingSplit, GenderDimension, AgeTNSDimension
from time import sleep
import time
import logging, os, ConfigParser
import json
import api_connector as facebook_api

__author__ = 'aalogachev'

# setting logging
MIN_BUDGET = 0  # 50*100
MIN_CPC_BID = 0  # 5*100
logger = logging.getLogger('ru.yandex.facebook.targeting_split')

# # reading config
# config = ConfigParser.SafeConfigParser()
# config.readfp(open(os.path.join(utils.get_project_path(), 'config_facebook_public.json')))
# section = config.defaults()['active_section']
# logger.debug('Using config section =' + section)
#
# # reading access parameters from config
# my_app_id = config.get(section=section, option='app_id')
# my_app_secret = config.get(section=section, option='app_secret')
# my_access_token = config.get(section=section, option='access_token')
# my_adv_account_id = config.get(section=section, option='adv_account_id')
# # NOTE here used id2 turkey account
# turkey_adv_account_id = config.get(section=section, option='yndx_turkey_account_id2')

MAX_RETRY_COUNT = 5
SLEEP_TIMEOUT = 300

insights_params = {
    'date_preset': 'last_7_days',
    'filtering': [
        # using only link clicks
        {'field': 'action_type', 'operator': 'IN', 'value': ['link_click']}
    ]
}


class AdSetSplit:
    ad_set = None

    def __init__(self, daily_budget, lifetime_budget, impressions, targeting_split):
        self.daily_budget = int(daily_budget)
        self.lifetime_budget = int(lifetime_budget)
        self.impressions = int(impressions)
        self.targeting_split = targeting_split


ad_campaign_special_fields = [
    AdCampaign.Field.id,
    AdCampaign.Field.name,
    AdCampaign.Field.objective,
    AdCampaign.Field.buying_type,
    AdCampaign.Field.adlabels,
]
ad_set_fields_to_copy = [
    AdSet.Field.adlabels,
    AdSet.Field.bid_amount,
    # AdSet.Field.bid_info,
    AdSet.Field.billing_event,
    AdSet.Field.campaign_schedule,
    AdSet.Field.creative_sequence,  ##TODO can perform test of this feature (no interface for this)
    AdSet.Field.end_time,
    AdSet.Field.is_autobid,
    AdSet.Field.optimization_goal,
    AdSet.Field.product_ad_behavior,
    AdSet.Field.pacing_type,  ##TODO can perform test of this feature (pacing type = schedule)
    AdSet.Field.promoted_object,
    AdSet.Field.rf_prediction_id,  ##TODO need to think here
    AdSet.Field.rtb_flag,
    AdSet.Field.status,

    ##campaign_group_id = 'campaign_group_id'
    ##created_time = 'created_time'
    ##updated_time = 'updated_time'
]

ad_set_special_fields = [
    AdSet.Field.id,
    AdSet.Field.daily_budget,  # need to set correct way (daily budget = split(daily budget))
    AdSet.Field.budget_remaining,  # no need to set for new adset
    AdSet.Field.lifetime_budget,  # need to set correct way (lifetime budget = split(lifetime budget))
    AdSet.Field.lifetime_imps,  # need to set correct way (lifetime imps = split(remaining impressions))
    AdSet.Field.name,  # need to set correctly (name = name + slice description)?
    AdSet.Field.targeting,  # new split should be added to targeting
    AdSet.Field.start_time,  # need to set correctly (start_time = now)
]

ad_group_fields_to_copy = [
    # FIXME think about additional fields
    # AdGroup.Field.adgroup_review_feedback = 'adgroup_review_feedback' # think about it!
    AdGroup.Field.adlabels,
    AdGroup.Field.bid_amount,
    # bid_info = 'bid_info' #no set method for this field in v2.4
    # campaign_group_id = 'campaign_group_id' #no set method for this field
    # AdGroup.Field.campaign_id = 'campaign_id' #no need to copy this field
    # AdGroup.Field.conversion_specs, # deprecated for v2.4
    # created_time = 'created_time'
    # AdGroup.Field.creative,
    # AdGroup.Field.failed_delivery_checks #think about it!
    AdGroup.Field.name,
    # redownload = 'redownload' #think about it!
    AdGroup.Field.social_prefs,
    AdGroup.Field.tracking_specs,
    # updated_time = 'updated_time'
    # AdGroup.Field.view_tags, #TODO "message": "(#3) App must be on whitelisted for View tags access",
]

ad_group_special_fields = [
    AdGroup.Field.id,
    AdGroup.Field.creative,
    AdGroup.Field.status,
]

ad_creative_fields_to_copy = [
    # AdCreative.Field.actor_id,
    # AdCreative.Field.actor_image_hash,
    # AdCreative.Field.actor_name,
    AdCreative.Field.adlabels,
    # AdCreative.Field.applink_treatment,
    AdCreative.Field.body,
    AdCreative.Field.call_to_action_type,
    # AdCreative.Field.filename,
    # AdCreative.Field.follow_redirect,
    # AdCreative.Field.id,
    # AdCreative.Field.image_crops,
    # AdCreative.Field.image_file,
    AdCreative.Field.image_hash,
    # AdCreative.Field.image_url,
    AdCreative.Field.link_deep_link_url,
    AdCreative.Field.link_url,
    AdCreative.Field.name,
    # AdCreative.Field.object_id,
    AdCreative.Field.object_store_url,
    # AdCreative.Field.object_story_id,
    AdCreative.Field.object_story_spec,
    AdCreative.Field.object_type,
    AdCreative.Field.object_url,
    # preview_url = 'preview_url'
    # product_set_id = 'product_set_id'
    # template_url = 'template_url'
    # thumbnail_url = 'thumbnail_url'
    AdCreative.Field.title,
    # video_id = 'video_id'
]
ad_creative_special_fields = [
    AdCreative.Field.id,
    AdCreative.Field.url_tags,
]


def get_split_dimension(adset):
    """
    Determines which dimension to use while splitting
    :type adset: AdSet
    """
    # FIXME analyze insights to get split
    return None


def split_budget_and_impressions(adset, dimensions_to_split):
    res = []
    all_splits = []
    # generating cross product of splits
    for dimension in dimensions_to_split:
        new_splits = dimension.get_targeting_splits()
        new_intersections = []
        if (len(all_splits) == 0):
            new_intersections = new_splits
        else:
            for split1 in all_splits:
                for split2 in new_splits:
                    new_intersections.append(split1.intersect(split2))
        all_splits = new_intersections
    # all_splits = dimension_to_split.get_targeting_splits()
    # budget_remaining = int(adset.get(AdSet.Field.budget_remaining))
    lifetime_budget = int(adset.get(AdSet.Field.lifetime_budget))
    impressions_remaining = int(adset.get(AdSet.Field.lifetime_imps))
    daily_budget = int(adset.get(AdSet.Field.daily_budget))
    num_of_splits = len(all_splits)
    for i in range(0, num_of_splits):
        # FIXME add right formula for splitting money based on insights (this formula has error)
        res.append(
            AdSetSplit(
                daily_budget * ((i + 1) / num_of_splits) - daily_budget * (i / num_of_splits),
                lifetime_budget,
                impressions_remaining * ((i + 1) / num_of_splits) - impressions_remaining * (i / num_of_splits),
                all_splits[i]
            ))
    return res


def split_by_dimension(adset, dimensions_to_split, ad_campaign, ad_account):
    """
    Splits adset by split_dimensions, copies all underlying objects (adgroups, adcreatives)
    :type adset: AdSet
    :type dimension_to_split: Dimension
    :type ad_campaign: AdCampaign
    :type ad_account: AdAccount
    """
    res = []

    #caching all labels for later user
    all_ad_labels = list(my_account.get_ad_labels())

    adset = remote_read_safe(adset, fields=ad_set_fields_to_copy + ad_set_special_fields)
    # FIXME add check targeting to possibility of split
    cur_targeting = adset.get(AdSet.Field.targeting)
    cur_name = adset.get(AdSet.Field.name)

    adgroups_from_adset = adset.get_ad_groups(fields=ad_group_fields_to_copy + ad_group_special_fields)
    adgroups_from_adset_fetched = []
    for ad_group in adgroups_from_adset:
        adgroups_from_adset_fetched.append(ad_group)

    adset_budget_splits = split_budget_and_impressions(adset, dimensions_to_split)
    for budget_split in adset_budget_splits:
        new_adset = clone_adset(adset, ad_account)
        assert isinstance(budget_split, AdSetSplit)
        targeting_split = budget_split.targeting_split
        labels = get_label_ids(targeting_split.labels, ad_account, all_ad_labels)
        url_tags = targeting_split.url_tags
        url_tags = url_tags[:]
        url_tags.append(u'cd_f_campaign_id={0}'.format(ad_campaign.get(AdCampaign.Field.id)))
        new_targeting = targeting_split.intersectWithTargeting(cur_targeting)
        if (new_targeting is None):
            logger.warning(
                u'No targeting {0} intersection with targeting split {1}'.format(cur_targeting, targeting_split))
            continue
        # new_targeting.update(targeting_split.targeting)
        # setting name, budget, impressions, start_time, new targeting and labels
        new_adset.update({
            AdSet.Field.daily_budget: max(budget_split.daily_budget, MIN_BUDGET),
            AdSet.Field.lifetime_budget: max(budget_split.lifetime_budget, MIN_BUDGET),
            # FIXME remove from here
            # AdSet.Field.bid_amount: MIN_CPC_BID,
            AdSet.Field.lifetime_imps: budget_split.impressions,
            AdSet.Field.targeting: new_targeting,
            AdSet.Field.name: u'{0} | {1}'.format(cur_name, u' | '.join(targeting_split.labels)),
            AdSet.Field.start_time: int(time.time()) + 15,
            AdSet.Field.campaign_group_id: ad_campaign.get_id_assured(),
            AdSet.Field.status: AdSet.Status.paused,  # FIXME remove all this after all will be done
        })
        remote_create_safe(new_adset)
        add_labels_safe(new_adset, labels)
        logger.info(u'AdSet created: {0}'.format(new_adset))
        # copying adgroups
        clone_and_create_adgroups(adgroups_from_adset_fetched, new_adset, labels,
                                  url_tags + [u'cd_f_adset_id={0}'.format(new_adset.get(AdSet.Field.id))], ad_account)
        logger.info(u'AdSet fully copied: {0}'.format(new_adset))
        budget_split.ad_set = new_adset
        res.append(budget_split)
    return res


def clone_and_create_adgroups(ad_groups, new_ad_set, labels, url_tags, ad_account):
    """
    Clones and remotely creates new AdGroups based on list of ad_groups prototypes
    :type ad_groups: list
    :type new_ad_set: AdSet
    :type labels: list
    :type url_tag: []
    :type ad_account: AdAccount
    """
    res = []

    ad_creative_batch = ad_account.get_api_assured().new_batch()

    def callback_failure(response):
        raise response.error()

    # one can try  using creative field
    # JSON object
    # Create an ad creative first, then reference the ad creative in this field by setting the value to {'creative_id':<ad_creative_id>}

    map_adgroup_id_creatives = dict()
    for ad_group in ad_groups:
        assert isinstance(ad_group, AdGroup)
        # no need to clone copy ad_group if it contains no ad_creatives
        ad_creatives = get_adcreatives_safe(ad_group, fields=ad_creative_fields_to_copy + ad_creative_special_fields)
        if (ad_creatives is None or len(ad_creatives) == 0):
            logger.info(u'AdGroup[{0},{1}] is skipped'.format(AdGroup.get_id(), AdGroup.get(AdGroup.Field.name)))
        else:
            # suppose it contain only one adcreative
            new_ad_creatives = clone_and_create_ad_creatives(ad_creatives, labels, url_tags, ad_account,
                                                             ad_creative_batch, callback_failure)
            map_adgroup_id_creatives[ad_group.get('id')] = new_ad_creatives

    bacth_execute_safe(ad_creative_batch)
    # all creatives copied so copiing

    ad_group_batch = ad_account.get_api_assured().new_batch()
    for ad_group in ad_groups:
        assert isinstance(ad_group, AdGroup)
        if (map_adgroup_id_creatives.has_key(ad_group.get('id'))):
            adcreatives = map_adgroup_id_creatives.get(ad_group.get('id'))
            # NOTE assuming only one creative per adgroup
            if (len(adcreatives) > 1): raise Exception(
                'Was assumed that only one creative per adgroup, but this assertion is false')
            new_ad_group = clone_adgroup(ad_group, ad_account)
            new_ad_group[AdGroup.Field.campaign_id] = new_ad_set.get_id_assured()
            new_ad_group[AdGroup.Field.status] = AdGroup.Status.paused
            new_ad_group[AdGroup.Field.creative] = {"creative_id": new_ad_creatives[0].get_id_assured()}
            new_ad_group.remote_create(ad_group_batch, failure=callback_failure)
            # remote_create_safe(new_ad_group)
            add_labels_safe(new_ad_creatives[0], labels)
            logger.info(u'AdGroup added to batch: {0}'.format(new_ad_group))
            res.append(new_ad_group)

    bacth_execute_safe(ad_group_batch)
    # all adgroups are created

    # adding labels for all adgroups
    for new_ad_group in res:
        add_labels_safe(new_ad_group, labels)

    return res


def clone_and_create_ad_creatives(ad_creatives, labels, url_tags, ad_account, api_batch, callback_failure):
    """
    Clone and creates ad creatives base on list of ad_creatives prototypes
    :type ad_creatives: list
    :type labels: list
    :type url_tag: []
    :type ad_account: AdAccount
    """
    # https://developers.facebook.com/bugs/1605648466391176/?comment_id=1039672816061177

    res = []

    #FIXME complete this code
    def callback_success(response):
        return
        #creative_id = -1
        #res.append(creative_id)
        #print response

    for ad_creative in ad_creatives:
        assert isinstance(ad_creative, AdCreative)
        logger.debug(u'Copying AdCreative: {0}'.format(ad_creative))
        new_ad_creative = clone_adcreative(ad_creative, ad_account)
        new_url_tags = ad_creative.get(AdCreative.Field.url_tags)
        if new_url_tags is None:
            new_url_tags = u'&'.join(url_tags)
        else:
            new_url_tags = u'{0}&{1}'.format(new_url_tags, u'&'.join(url_tags))
        new_ad_creative[AdCreative.Field.url_tags] = new_url_tags
        new_ad_creative.remote_update(batch=api_batch, failure=callback_failure, success=callback_success)
        # remote_create_safe(new_ad_creative)
        # add_labels_safe(new_ad_creative, labels)
        logger.info(u'AdCreative added to batch: {0}'.format(new_ad_creative))
        res.append(new_ad_creative)
    return res


def add_labels_safe(ad_object, ad_labels):
    if ad_labels is None or len(ad_labels) == 0:
        return
    retry_count = 0
    is_success = False
    last_exception = None
    while (not is_success) and retry_count < MAX_RETRY_COUNT:
        try:
            # calling facebook API to add labels to ad_object
            ad_object.add_labels(ad_labels)
            is_success = True
        except FacebookRequestError as e:
            logger.warning(u'Exception occur: {0}'.format(e))
            last_exception = e
            retry_count += 1
            # https://developers.facebook.com/docs/marketing-api/api-rate-limiting
            logger.warning(u'Sleeping for {0} seconds - trying to avoid rate-limiting'.format(SLEEP_TIMEOUT))
            sleep(SLEEP_TIMEOUT)
    if retry_count >= MAX_RETRY_COUNT:
        logger.error(u'Num of retries exceeded {0} with last exception: {1}'.format(MAX_RETRY_COUNT, last_exception))
        raise e


def bacth_execute_safe(api_batch):
    retry_count = 0
    is_success = False
    last_exception = None
    while (not is_success) and retry_count < MAX_RETRY_COUNT:
        try:
            api_batch.execute()
            is_success = True
        except FacebookRequestError as e:
            logger.warning(u'Exception occur: {0}'.format(e))
            last_exception = e
            retry_count += 1
            # https://developers.facebook.com/docs/marketing-api/api-rate-limiting
            logger.warning(
                u'Sleeping for {0} seconds - trying to avoid rate-limiting at ad account level'.format(SLEEP_TIMEOUT))
            sleep(SLEEP_TIMEOUT)
    if retry_count >= MAX_RETRY_COUNT:
        logger.error(u'Num of retries exceeded with last exception: {1}'.format(last_exception))
        raise e


def remote_create_safe(ad_object):
    retry_count = 0
    is_success = False
    last_exception = None
    while (not is_success) and retry_count < MAX_RETRY_COUNT:
        try:
            ad_object.remote_create()
            is_success = True
        except FacebookRequestError as e:
            logger.warning(u'Exception occur: {0}'.format(e))
            last_exception = e
            retry_count += 1
            # https://developers.facebook.com/docs/marketing-api/api-rate-limiting
            logger.warning(
                u'Sleeping for {0} seconds - trying to avoid rate-limiting at ad account level'.format(SLEEP_TIMEOUT))
            sleep(SLEEP_TIMEOUT)
    if retry_count >= MAX_RETRY_COUNT:
        logger.error(u'Num of retries exceeded with last exception: {1}'.format(last_exception))
        raise e


def remote_read_safe(ad_object, fields):
    retry_count = 0
    is_success = False
    last_exception = None
    while (not is_success) and retry_count < MAX_RETRY_COUNT:
        try:
            ad_object.remote_read(fields=fields)
            is_success = True
        except FacebookRequestError as e:
            logger.warning(u'Exception occur: {0}'.format(e))
            last_exception = e
            retry_count += 1
            # https://developers.facebook.com/docs/marketing-api/api-rate-limiting
            logger.warning(
                u'Sleeping for {0} seconds - trying to avoid rate-limiting at ad account level'.format(SLEEP_TIMEOUT))
            sleep(SLEEP_TIMEOUT)
    if retry_count >= MAX_RETRY_COUNT:
        logger.error(u'Num of retries exceeded with last exception: {1}'.format(last_exception))
        raise e
    return ad_object


def get_adcreatives_safe(ad_object, fields):
    retry_count = 0
    is_success = False
    last_exception = None
    while (not is_success) and retry_count < MAX_RETRY_COUNT:
        try:
            creatives = ad_object.get_ad_creatives(fields=fields)
            is_success = True
        except FacebookRequestError as e:
            logger.warning(u'Exception occur: {0}'.format(e))
            last_exception = e
            retry_count += 1
            # https://developers.facebook.com/docs/marketing-api/api-rate-limiting
            logger.warning(
                u'Sleeping for {0} seconds - trying to avoid rate-limiting at ad account level'.format(SLEEP_TIMEOUT))
            sleep(SLEEP_TIMEOUT)
    if retry_count >= MAX_RETRY_COUNT:
        logger.error(u'Num of retries exceeded with last exception: {1}'.format(last_exception))
        raise e
    return creatives


def clone_adset(ad_set, ad_account):
    """
    Instantiate new AdSet and copies all copy_fields from ad_set prototype.
    Does not create remote object
    :type ad_set: AdSet
    :type ad_account: AdAccount
    """
    new_ad_set = AdSet(parent_id=ad_account.get_id_assured())
    update_res = {}
    for field in ad_set_fields_to_copy:
        if ad_set.get(field) is not None:
            update_res[field] = ad_set.get(field)
    ##budget/impression splitting, naming, targeting will be performed outside this method
    new_ad_set.update(update_res)
    ##write of ad_set will be performed outside this method
    return new_ad_set


def clone_adgroup(ad_group, ad_account):
    """
    Instantiate new AdGroup and copies all copy_fields from ad_group prototype.
    Does not create remote object
    :type ad_group: AdGroup
    :type ad_account: AdAccount
    """
    new_ad_group = AdGroup(parent_id=ad_account.get_id_assured())
    update_res = {}
    for field in ad_group_fields_to_copy:
        if ad_group.get(field) is not None:
            update_res[field] = ad_group.get(field)
    new_ad_group.update(update_res)
    ##write of ad_group will be performed outside this method
    return new_ad_group


def clone_adcreative(ad_creative, ad_account):
    """
    Instantiate new AdCreative and copies all copy_fields from ad_creative prototype.
    Does not create remote object
    :type ad_creative: AdCreative
    :type ad_account: AdAccount
    """
    new_ad_creative = AdCreative(parent_id=ad_account.get_id_assured())
    update_res = {}
    for field in ad_creative_fields_to_copy:
        if ad_creative.get(field) is not None:
            update_res[field] = ad_creative.get(field)
    new_ad_creative.update(update_res)
    ##write of ad_creative will be performed outside this method
    return new_ad_creative


def create_ad_campaign(ad_campaign, split_dimensions, ad_account):
    """
    Remotely creates new AdCampaign from ad_campaign prototype
    :type ad_campaign: AdCampaign
    :type split_dimension: Dimension
    :type ad_account: AdAccount
    """
    ad_campaign = remote_read_safe(ad_campaign, ad_campaign_special_fields)
    ### Create a Campaign
    new_ad_campaign = AdCampaign(parent_id=ad_account.get_id_assured())
    campaign_ad_labels = ad_campaign.get(AdCampaign.Field.adlabels)
    if campaign_ad_labels is None: campaign_ad_labels = []
    new_campaign_postfix = u''
    for dimension in split_dimensions:
        new_campaign_postfix += u' | {0}'.format(dimension.keyword)
    new_ad_campaign[AdCampaign.Field.name] = u'{0} | {1}{2} '.format(
        ad_campaign.get(AdCampaign.Field.name),
        ad_campaign.get(AdCampaign.Field.id),
        new_campaign_postfix),
    new_ad_campaign[AdCampaign.Field.objective] = ad_campaign.get(AdCampaign.Field.objective)
    new_ad_campaign[AdCampaign.Field.buying_type] = ad_campaign.get(AdCampaign.Field.buying_type)
    new_ad_campaign[AdCampaign.Field.status] = AdCampaign.Status.paused
    new_ad_campaign[AdCampaign.Field.adlabels] = ad_campaign.get(AdCampaign.Field.adlabels)

    remote_create_safe(new_ad_campaign)

    # adding new adlabels to adcampaign
    # FIXME add labels for dimension spliting here
    # dimension_label_id = get_adlabel_id(u'split_{0}'.format(split_dimension.keyword), ad_account)
    # add_labels_safe(new_ad_campaign, [dimension_label_id])
    logger.info(u'Campaign created: {0}'.format(new_ad_campaign))
    return new_ad_campaign


def get_label_ids(labels, ad_account, all_ad_labels):
    res = []
    for label in labels:
        res.append(get_adlabel_id(label, ad_account, all_ad_labels))
    return res


def get_adlabel_id(label, ad_account, all_ad_labels):
    label_id = None
    for ad_label in all_ad_labels:
        if ad_label.get(AdLabel.Field.name) == label:
            label_id = ad_label.get(AdLabel.Field.id)
    if label_id is None:
        new_adlabel = AdLabel(parent_id=ad_account.get_id_assured())
        new_adlabel[AdLabel.Field.name] = label
        remote_create_safe(new_adlabel)
        all_ad_labels.append(new_adlabel)
        label_id = new_adlabel.get_id_assured()
    return label_id


def run_all_for_adset(ad_account, campaign_id, adset_id, split_dimensions):
    """
    Runs splitting procedure for adset in adcampaign
    :type ad_account: AdAccount
    """

    ad_campaign_proto = AdCampaign(campaign_id)
    adset = AdSet(adset_id)

    new_campaign = create_ad_campaign(ad_campaign_proto, split_dimensions, ad_account)
    split_by_dimension(adset, split_dimensions, new_campaign, ad_account)


if __name__ == '__main__':
    #initializing logging
    logging.basicConfig(format='%(asctime)s : %(levelname)s : %(message)s', level=logging.DEBUG)
    #initializing facebook api and getting account
    my_account = facebook_api.ApiConnector().get_account()
    logger.info(u'Using account = ' + str(my_account))
    logger.info('********** Ad splitting begins. **********')
    split_dimensions = [GenderDimension(), AgeTNSDimension()]

    #Experiment2 Turkish Campaign
#    run_all_for_adset(my_account, 6026893966608, 6026893967608, split_dimensions)

    # my test campaign
    run_all_for_adset(my_account, 6029373108432, 6029373109232, split_dimensions)


