$(function() {
    _ = require("underscore");
    var Backbone = require("backbone");
    var Marionette = require("backbone.marionette");
    
    var Handlebars = window.Handlebars = require("handlebars/runtime");
    Handlebars.registerHelper('prettyJson', function(j) {
            return JSON.stringify(j, null, '  '); j.replace(/^{\n*/, '').replace('\n}', '');
    });

    var CandidateList = Backbone.Model.extend({
    });

    var CandidateListCollection = Backbone.Collection.extend({
        model: CandidateList,
        comparator: function(a, b) {
            if (a.get('my') != b.get('my')) {
                return a.get('my') < b.get('my');
            }
            return a.get('id') < b.get('id');
        },
        url: '/atom/api/v1/lists/'
    });

    var CandidateListView = Marionette.ItemView.extend({
        model: CandidateList,
        tagName: "tr",
        onRender: function(view) {
            $(view.el).find("input").bootstrapToggle();
        },
        template: require('./clist.hbs'),
        ui: {toggle: ".toggleCB"},
        events: {
            "change .toggle-group": "toggleList"
        },
        toggleList: function(view) {
            alert("toggleList!");
        }

    });

    var ListsView = Marionette.CollectionView.extend({
        tagName: "table",
        className: "table table-hover",
        childView: CandidateListView,
    });


    function get_var(path, object) {
        var i, names = path.split(".");
        for (i = 0; i < names.length; i++) {
            object = object[names[i]];
            if (!object)
                return "";
        }
        return object;
    }

    var filter_eval = function(expr, object) {
        if (expr[0] == "=") {
            return get_var(expr[1], object) == expr[2];
        } else if (expr[0] == "~") {
            return get_var(expr[1], object).indexOf(expr[2]) >= 0;
        } else if (expr[0] == "!~") {
            return get_var(expr[1], object).indexOf(expr[2]) < 0;
        } else if (expr[0] == "!=") {
            return get_var(expr[1], object) != expr[2];
        } else if (expr[0] == "<") {
            return Number(get_var(expr[1], object)) < Number(expr[2]);
        } else if (expr[0] == ">") {
            return Number(get_var(expr[1], object)) > Number(expr[2]);
        } else if (expr[0] == "&") {
            var i;
            for (i = 1; i < expr.length; i++) {
                if (!filter_eval(expr[i], object))
                    return false;
            }
            return true;
        } else if (expr[0] == "|") {
            var i;
            for (i = 1; i < expr.length; i++) {
                if (filter_eval(expr[i], object))
                    return true;
            }
            return false;
        }
    };

    var filter_parser = require("./filter.pegjs");

    // table of candidates

    var Candidate = Backbone.Model.extend({
        defaults: { selected: true, production_list: window.atom_production_list },

        initialize: function(options) {
            this.options = _.extend(this.defaults, this.options);
        }
    });
    var CandidateCollection = Backbone.Collection.extend({
        initialize: function(models, options) {
            this.slug = options.slug;
        },
        model: Candidate,
        url: function() {return '/atom/api/v1/lists/' + this.slug}
    });
    
    var CandidateView = Marionette.ItemView.extend({
        model: Candidate,
        tagName: "div",
        className: "checkbox",
        ui: {
            checkbox: "input[type=checkbox]"
        },
        events: {
            "click @ui.checkbox": "toggleSelection"
        },
        modelEvents: {
            "change": "render"
        },
        toggleSelection: function() {
            this.model.set('selected', !this.model.get('selected'));
        },
        serializeData: function() { return this.model.toJSON(); },
        template: require('./candidate.hbs')
    });

    var Tristate = Backbone.Model.extend({
        defaults: {
            state: 'on',
        },
        isOn: function() {
            return this.get('state') == "on";
        },
    });
    
    var ViewSettings = Backbone.Model.extend({
        defaults: {
            select_all: "on",
            filterString: ""
        },
        allSelected: function() {
            return this.get('select_all') == 'on';
        },
        selectionIndeterminate: function() {
            return this.get('select_all') == "indeterminate";
        },
        setIndeterminate: function() {
            this.set('select_all', 'indeterminate');
        },
        toggleSelectAll: function() {
            if (this.allSelected() || this.selectionIndeterminate()) {
                this.set('select_all', 'off');
            } else {
                this.set('select_all', 'on');
            }
        }
    });

    var SpinnerView = Marionette.ItemView.extend({
        template: require('./empty.hbs')
    });

    var CandidatesView = Marionette.CompositeView.extend({
        initialize: function() {
            this.model = new ViewSettings();
        },
        el: '#application',
        template: require('./layout.hbs'),
        childViewContainer: '#candidates',
        childView: CandidateView,
        emptyView: SpinnerView,
        ui: {
            select_all: "#select_all",
            sort_buttons: "span.sort_by"
        },
        events: {
            "click @ui.select_all": "toggleSelectAll",
            "click @ui.sort_buttons": "onSort"
        },
        collectionEvents: {
            "indeterminateSelection": "indSel"
        },
        onRender: function() {
            if (this.model.selectionIndeterminate()) {
                $("#select_all").prop('indeterminate', true);
            }
        },
        firstRender: function() {
            this.render();
        },
        updateFilter: function(pat) {
            this.model.set('filterString', pat, {silent: true});
            if (pat.length < 1) {
                this.filter = null;
                this.render();
                return;
            }

            try {
                var cf = filter_parser.parse(pat);
            } catch (e) {
                console.log('bad filter expression [' + pat + ']');
                return;
            }
            console.log('setting filter: ' + JSON.stringify(cf));

            this.filter = function(model) {
                return filter_eval(cf, model.get('json'));
            };
            this.render();
        },
        toggleSelectAll: function() {
            this.model.toggleSelectAll();
            var newState = this.model.allSelected();
            this.children.each(function(childView) {
                childView.model.set('selected', newState);
            });
        },
        indSel: function() {
            this.model.setIndeterminate();
            this.render();
        },
        setSortBy: function(field) {
           if (field.indexOf("st.") == 0) {
               this.collection.comparator = function(model) {
                   return 1.0 - _.max([0, model.get('st')[field.substr(3)]]);
               };
            } else {
               this.collection.comparator = function(model) {
                   return model.get('json')[field];
               };
            }   
            this.collection.sort();
        },
        onSort: function(evt) {
        }
    });

    var AtomAdmin = Marionette.Application.extend({
        initialize: function() {
            this.collection = new CandidateCollection(window.preloaded_data, {slug: window.atom_admin_slug});
            this.rootView = new CandidatesView({collection: this.collection});
        },

        candidates: function(cid) {
            this.rootView.render();
        },

        setFilter: function(pat) {
        },

    });

    var app = new AtomAdmin();
    window.atom_admin = app;
    
    var router = new Marionette.AppRouter({
        appRoutes: {
            '': 'candidates',
            'collections/:id': 'candidates'
        },
        controller: app
    });

    app.on('start', function() {
        Backbone.history.start();
    });


    app.start();

    $("#candidate_search_button").click(function() {
        var pat = $("#candidate_search").val();
        app.rootView.updateFilter(pat);
    });
    
    $("span.sort_by").click(function() {
        var elt = $(this);
        $("span.sort_by").removeClass("sortKey");
        elt.addClass("sortKey");
        app.rootView.setSortBy(elt.data('field'));
    });
});

// vim:set backupcopy=yes et ts=4 sw=4:
