from __future__ import print_function

import os
import sys
import json
import jsonpickle
import logging
import traceback

import yt.logger

logger = logging.getLogger(__name__)


def remap_fd(fp, path):
    with open(path, "w") as new_fp:
        old_fp = os.fdopen(os.dup(fp.fileno()), "w")
        os.close(fp.fileno())
        os.dup2(new_fp.fileno(), fp.fileno())

    return old_fp


def emit_record(stream, type, value):
    record = {
        "type": type,
        "value": value
    }

    stream.write("%s\n" % json.dumps(record))


class JsonStreamHandler(logging.StreamHandler):
    def makePickle(self, record):
        """
        See logging.SocketHandler
        """
        ei = record.exc_info
        if ei:
            self.format(record)
            record.exc_info = None

        d = dict(record.__dict__)
        d["msg"] = record.getMessage()
        d["args"] = None
        s = jsonpickle.dumps(d)

        if ei:
            record.exc_info = ei

        return s

    def emit(self, record):
        try:
            msg = self.makePickle(record)
            emit_record(self.stream, "log", msg)
            self.flush()
        except (KeyboardInterrupt, SystemExit):
            raise
        except:
            self.handleError(record)


def setup_logging(stream):
    fmt = logging.Formatter(logging.BASIC_FORMAT)

    handler = JsonStreamHandler(stream=stream)
    handler.setFormatter(fmt)

    root = logging.getLogger()
    root.setLevel(logging.DEBUG)
    root.addHandler(handler)

    yt.logger.LOGGER.addHandler(handler)


def main():
    stderr = remap_fd(sys.stderr, "stderr")

    def excepthook(etype, value, tb):
        lines = traceback.format_exception(etype, value, tb)
        for line in lines:
            emit_record(stderr, "log", line)

    sys.excepthook = excepthook

    setup_logging(stderr)

    if len(sys.argv) < 2:
        print("Usage: {} block_spec".format(sys.argv[0]))
        sys.exit(1)

    with open(sys.argv[1]) as fp:
        request_dict = json.load(fp)

    from google.protobuf import json_format

    from tasklet.api import tasklet_pb2
    from tasklet.runtime import dispatch

    request = tasklet_pb2.JobInstance()
    json_format.ParseDict(request_dict, request)
    request_data = request.SerializeToString()

    response_data = dispatch.dispatch(request_data)

    response = tasklet_pb2.JobResult()
    response.ParseFromString(response_data)
    result = json_format.MessageToJson(response)

    emit_record(stderr, "result", result)


if __name__ == "__main__":
    main()
