# coding=utf-8
import json
import logging
import os.path
import pathlib
import shlex
import subprocess
import tempfile

from google.protobuf import descriptor_pb2


logger = logging.getLogger(__name__)

PROTO_REGISTRY_BUILD_RESULT = "tasklet/registry/common/tasklet-registry-common.protodesc"


def locate_arc_root_by_t_yaml(path: pathlib.Path) -> pathlib.Path:
    path = path.absolute()
    assert path.exists()

    for parent in path.parents:
        if parent.joinpath(".arc", "HEAD").exists():
            return parent
    raise EnvironmentError("can not locate arc root")


def relative_arc_path_by_t_yaml(path: pathlib.Path) -> pathlib.Path:
    path = path.absolute()
    assert path.exists()

    for parent in path.parents:
        if parent.joinpath(".arc", "HEAD").exists():
            return path.relative_to(parent)

    return path


def is_t_yaml_locate_in_arc(path: pathlib.Path) -> bool:
    path = path.absolute()
    assert path.exists()

    for parent in path.parents:
        if parent.joinpath(".arc", "HEAD").exists():
            return True
    return False


def _compile_fds(arc_root: pathlib.Path) -> bytes:
    registry_path = os.path.dirname(arc_root.joinpath(PROTO_REGISTRY_BUILD_RESULT))
    with tempfile.TemporaryDirectory() as tmp_output_dir:
        result_path = pathlib.Path(tmp_output_dir).joinpath(PROTO_REGISTRY_BUILD_RESULT)
        cmd = [
            "./ya",
            "make",
            "--output", tmp_output_dir,
            registry_path,
        ]
        logger.info(f"Building FDS. Cmd: \"{shlex.join(cmd)}\", Cwd: \"{str(arc_root)}\"")
        rc = subprocess.check_call(cmd, cwd=str(arc_root))
        assert rc == 0, f"Ya make failed with code {rc}"
        assert result_path.exists(), f"Output does not exist: {result_path}"
        return result_path.read_bytes()


def build_registry_fds(arc_root: pathlib.Path) -> descriptor_pb2.FileDescriptorSet:
    logger.debug("Building schema registry...")
    fds_bytes = _compile_fds(arc_root)
    logger.debug("Compiled schema data size: %sb", len(fds_bytes))

    fds = descriptor_pb2.FileDescriptorSet()
    fds.ParseFromString(fds_bytes)
    return fds


def get_arc_info(arc_root: pathlib.Path) -> dict:
    rv = subprocess.check_output(
        ["arc", "info", "--json"], cwd=str(arc_root)
    )
    return json.loads(rv)
