# coding=utf-8
import json
import pathlib
import socket
import typing

import click
import inject

from tasklet.experimental.cli import arcadia_support
from tasklet.experimental.cli import consts
from tasklet.experimental.cli import context
from tasklet.experimental.cli import interfaces
from tasklet.experimental.cli import sandbox
from tasklet.experimental.cli import tasklet_descriptor
import tasklet.api.v2.data_model_pb2 as data_model
import tasklet.api.v2.tasklet_service_pb2 as tasklet_service


@click.group(name="build", cls=context.GroupWithContextOptions, help="List, create or get tasklet builds")
@context.option_cluster
@context.option_output
def build_subcommand():
    pass


@build_subcommand.command(name="list", help="Show list of tasklet builds")
@context.tasklet_descriptor_argument(argument_type=context.TaskletArgumentType.TASKLET)
@click.option(
    "-l", "--limit", metavar="<limit>", type=click.IntRange(1), default=100, show_default=True,
    help="Max number of builds to show."
)
@click.option(
    "-pt", "--page-token", metavar="<page>", type=click.IntRange(0), default=0, show_default=True,
    help="Page token to start show with."
)
def list_builds(namespace: str, tasklet: str, limit: int, page_token: int):
    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)
    result: tasklet_service.ListBuildsResponse = tasklet_service.ListBuildsResponse(token=page_token)
    if page_token == 1:
        click.echo("Wrong page token '1', it is used as final page", err=True)
        return
    while limit > 0:
        page_size = min(limit, 50)
        response: tasklet_service.ListBuildsResponse = ctx.execute_request(
            ctx.driver.get_tasklet_client().ListBuilds,
            tasklet_service.ListBuildsRequest(namespace=namespace, tasklet=tasklet, limit=page_size, token=result.token)
        )
        if response.builds:
            result.builds.extend(response.builds[:limit])
            limit -= page_size
            if response.token == 1:
                result.token = response.token
                break
        elif result.token == 0:
            click.echo(f"No builds in tasklet '{namespace}/{tasklet}'", err=True)
            return
        result.token = response.token
    ctx.dump_proto_message(result)
    if ctx.output == consts.Output.TABLE:
        click.echo(f"page token: {result.token}")


@build_subcommand.command(name="get", help="Get build by ID")
@click.argument("build_id")
def get_build(build_id: str) -> data_model.Build:
    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)
    response: tasklet_service.GetBuildResponse = ctx.execute_request(
        ctx.driver.get_tasklet_client().GetBuild,
        tasklet_service.GetBuildRequest(build_id=build_id),
    )
    ctx.dump_proto_message(response)
    return response.build


@build_subcommand.command(name="create", help="Register new build using resource ID (t.yaml required)")
@context.tasklet_descriptor_argument
# TODO:
#  * check if tasklet already exists (or create it if not)
#  * (optional) create or move label on current fresh build
@click.argument("resource_id", metavar="<resource_id>", type=int)
@click.option("--schema-id", metavar="<schema_id>", type=click.STRING)
@click.option("--description", metavar="<text>", type=click.STRING)
def create_build(resource_id: int, schema_id: typing.Optional[str], description: typing.Optional[str]):
    create_build_impl(resource_id=resource_id, schema_id=schema_id, description=description)


def create_build_impl(resource_id: int, schema_id: typing.Optional[str], description: typing.Optional[str]):
    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)
    # noinspection PyTypeChecker
    config: tasklet_descriptor.TaskletDescriptor = inject.instance(interfaces.ITaskletDescriptor)

    if schema_id:
        if config.state.spec.naive_schema.schema_id == "":
            config.state.spec.naive_schema.schema_id = schema_id
        if schema_id != config.state.spec.naive_schema.schema_id:
            click.echo(
                f"Conflicting schema_id. Command line: {schema_id}, Discovered: "
                f"{config.state.spec.naive_schema.schema_id}",
                err=True,
            )
            exit(1)
    if config.state.spec.naive_schema.schema_id == "":
        click.echo("Missing schema id", err=True)
        exit(1)
    response: tasklet_service.CreateBuildResponse = ctx.execute_request(
        ctx.driver.get_tasklet_client().CreateBuild,
        config.make_create_build_request(
            resource_id,
            description if description else "",
        )
    )
    ctx.dump_proto_message(response)


@build_subcommand.command(name="upload", help="Register new build using local file (t.yaml required)")
@context.tasklet_descriptor_argument
@click.option("--schema-id", metavar="<schema_id>", type=click.STRING)
@click.option("--build-schema", is_flag=True)
@click.option("--build-description", metavar="<text>", type=click.STRING)
@sandbox.upload_arguments
def upload_build(
    schema_id: typing.Optional[str],
    build_schema: bool,
    build_description: typing.Optional[str],
    description: str,
    attributes: [str],
    arch: str,
    resource_type: str,
    owner: str,
    file_meta: "common_upload.MDSHandle.FileMeta",
):
    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)

    if build_schema and schema_id:
        click.echo("--build-schema and --schema-id are mutually exclusive")
        exit(1)

    arc_info = {}
    if build_schema:
        # noinspection PyTypeChecker
        td: tasklet_descriptor.TaskletDescriptor = inject.instance(interfaces.ITaskletDescriptor)
        arc_root = arcadia_support.locate_arc_root_by_t_yaml(pathlib.Path(td.state.t_yaml_path))

        arc_info = arcadia_support.get_arc_info(arc_root)
        click.echo(f"Repository info: \n{json.dumps(arc_info, sort_keys=True, indent=4)}")

        click.echo("Building schema registry...")
        fds = arcadia_support.build_registry_fds(arc_root)
        click.echo("Registering schema...")
        schema_id = ctx.register_schema(fds, {"arc": arc_info}).hash
        click.echo(f"Registered schema with ID: {schema_id}")

    resource_id = sandbox.upload_resource(ctx.token, description, attributes, arch, resource_type, owner, file_meta)

    if build_description is None:
        mid = ""
        if arc_info.get("hash", ""):
            mid = f" Rev: {arc_info['hash']}"
        build_description = f"User upload.{mid} From: {socket.getfqdn()}"
    create_build_impl(resource_id=resource_id, schema_id=schema_id, description=build_description)
