# coding=utf-8
from typing import Optional

import click
import inject

from tasklet.experimental.cli import consts
from tasklet.experimental.cli import context
from tasklet.experimental.cli import interfaces
import tasklet.api.v2.data_model_pb2 as data_model
import tasklet.api.v2.tasklet_service_pb2 as tasklet_service


@click.group(name="label", cls=context.GroupWithContextOptions, help="List, create, get or move tasklet labels")
@context.option_cluster
@context.option_output
def label_subcommand():
    pass


@label_subcommand.command(name="list", help="Show list of tasklet labels")
@context.tasklet_descriptor_argument(argument_type=context.TaskletArgumentType.TASKLET)
def list_labels(namespace: str, tasklet: str):
    list_labels_impl(namespace=namespace, tasklet=tasklet)


def list_labels_impl(namespace: str, tasklet: str):
    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)

    labels_response: tasklet_service.ListLabelsResponse = ctx.execute_request(
        ctx.driver.get_tasklet_client().ListLabels,
        tasklet_service.ListLabelsRequest(namespace=namespace, tasklet=tasklet),
    )
    if not labels_response.labels:
        click.echo(f"No labels in tasklet '{tasklet}'", err=True)

    if ctx.output == consts.Output.TABLE and not ctx.quiet:
        from tasklet.experimental.cli import pretty_dumps

        ctx.quiet = True
        builds = []
        for label in labels_response.labels:
            if not label.spec.build_id:
                builds.append(None)
                continue
            response: tasklet_service.GetBuildResponse = ctx.execute_request(
                ctx.driver.get_tasklet_client().GetBuild,
                tasklet_service.GetBuildRequest(build_id=label.spec.build_id),
            )
            builds.append(response.build)
        extended_label_table = pretty_dumps.merge_tables(
            pretty_dumps.LabelDumper.dump_objects_list(labels_response.labels),
            pretty_dumps.BuildDumper.dump_objects_list(builds)
        )
        click.echo(extended_label_table.get_string())
    else:
        ctx.dump_proto_message(labels_response)


@label_subcommand.command(name="get", help="Get label info")
@context.tasklet_descriptor_argument(argument_type=context.TaskletArgumentType.LABEL)
def get_label(namespace: str, tasklet: str, label: str):
    get_label_impl(namespace=namespace, tasklet=tasklet, label=label)


def get_label_impl(namespace: str, tasklet: str, label: str) -> data_model.Label:
    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)
    response: tasklet_service.GetLabelResponse = ctx.execute_request(
        ctx.driver.get_tasklet_client().GetLabel,
        tasklet_service.GetLabelRequest(namespace=namespace, tasklet=tasklet, label=label)
    )
    ctx.dump_proto_message(response)
    return response.label


@label_subcommand.command(name="create", help="Create new label for tasklet")
@context.tasklet_descriptor_argument(argument_type=context.TaskletArgumentType.LABEL)
@click.option("-b", "--build", "build_id", metavar="<build_id>", help="Assign build to new label.")
def create_label(namespace: str, tasklet: str, label: str, build_id: Optional[str]):
    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)
    response: tasklet_service.CreateLabelResponse = ctx.execute_request(
        ctx.driver.get_tasklet_client().CreateLabel,
        tasklet_service.CreateLabelRequest(
            name=label,
            tasklet=tasklet,
            namespace=namespace,
            build_id=build_id,
        ),
    )
    ctx.dump_proto_message(response)


@label_subcommand.command(name="move", help="Move label between builds")
@context.tasklet_descriptor_argument(argument_type=context.TaskletArgumentType.LABEL)
@click.option("--from", "from_build", metavar="<build_id>", required=False, help="Previous build id (optional).")
@click.option("--to", "to_build", metavar="<build_id>", required=True, help="New build id.")
def move_label(namespace: str,
               tasklet: str,
               label: str,
               from_build: Optional[str],
               to_build: str):
    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)
    if from_build is None:
        ctx.quiet, quiet = True, ctx.quiet
        label_message = get_label_impl(namespace=namespace, tasklet=tasklet, label=label)
        ctx.quiet = quiet
        from_build = label_message.spec.build_id or None
    response: tasklet_service.MoveLabelResponse = ctx.execute_request(
        ctx.driver.get_tasklet_client().MoveLabel,
        tasklet_service.MoveLabelRequest(
            namespace=namespace,
            tasklet=tasklet,
            label=label,
            old_build_id=from_build,
            new_build_id=to_build,
        )
    )
    ctx.dump_proto_message(response)
