# coding=utf-8
from typing import Optional

import click
import inject

from tasklet.experimental.cli import consts
from tasklet.experimental.cli import context
from tasklet.experimental.cli import cmd_tasklet
from tasklet.experimental.cli import interfaces
import tasklet.api.v2.tasklet_service_pb2 as tasklet_service


@click.group(
    name="namespace",
    cls=context.GroupWithContextOptions,
    invoke_without_command=True,
    help="List, create or get namespaces"
)
@click.option("-O", "--owner", metavar="<owner>", help="Filter namespaces by owner.")
@context.option_cluster
@context.option_output
@click.pass_context
def namespace_subcommand(ctx: click.Context, owner: Optional[str]):
    if ctx.invoked_subcommand is None:
        list_namespaces_impl(owner)


@namespace_subcommand.command(name="list", help="List available namespaces")
@click.option("--owner", metavar="<owner>", help="Filter namespaces by owner.")
def list_namespaces(owner: Optional[str]):
    list_namespaces_impl(owner)


def list_namespaces_impl(owner: Optional[str]):
    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)

    response: tasklet_service.ListNamespacesResponse = ctx.execute_request(
        ctx.driver.get_tasklet_client().ListNamespaces, tasklet_service.ListNamespacesRequest(owner=owner)
    )
    if response.namespaces:
        ctx.dump_proto_message(response)
    elif owner:
        click.echo(f"No namespaces owned by '{owner}'", err=True)
    else:
        click.echo("No namespaces", err=True)


@namespace_subcommand.command(name="get", help="Get namespace by name")
@context.tasklet_descriptor_argument(argument_type=context.TaskletArgumentType.NAMESPACE)
def get_namespace(namespace: str):
    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)
    response: tasklet_service.GetNamespaceResponse = ctx.execute_request(
        ctx.driver.get_tasklet_client().GetNamespace, tasklet_service.GetNamespaceRequest(namespace=namespace)
    )
    ctx.dump_proto_message(response)
    if ctx.output == consts.Output.TABLE:
        cmd_tasklet.list_tasklets_impl(namespace=namespace)


@namespace_subcommand.command(name="create", help="Create new namespace (if not exists)")
@context.tasklet_descriptor_argument(argument_type=context.TaskletArgumentType.NAMESPACE)
@click.option("--owner", metavar="<owner>", required=True, help="Owner account id.")
def create_namespace(namespace: str, owner: Optional[str]):
    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)

    response: tasklet_service.CreateNamespaceResponse = ctx.execute_request(
        ctx.driver.get_tasklet_client().CreateNamespace,
        tasklet_service.CreateNamespaceRequest(
            name=namespace,
            account_id=owner,
        )
    )
    ctx.dump_proto_message(response)
