# coding=utf-8
import click
import pathlib
import typing
import json
import inject

import google.protobuf.message
from google.protobuf import descriptor_pb2

import tasklet.api.v2.schema_registry_service_pb2 as schema_service

from tasklet.experimental.cli import consts
from tasklet.experimental.cli import context
from tasklet.experimental.cli import interfaces


@click.group(
    name="sr",
    cls=context.GroupWithContextOptions,
    invoke_without_command=False,
    help="Manage schema registry"
)
@context.option_cluster
@context.option_output
def schema_registry_subcommand():
    pass


@schema_registry_subcommand.command(name="get", help="Get schema")
@click.argument("schema_id", type=click.STRING)
@click.argument(
    "output_path",
    metavar="<file path>",
    type=click.Path(path_type=pathlib.Path),
)
def get_schema(schema_id: str, output_path: pathlib.Path):
    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)
    req = schema_service.GetSchemaRequest(
        hash=schema_id,
    )

    client = ctx.driver.get_schema_registry_client()

    response: schema_service.GetSchemaResponse = ctx.execute_request(client.GetSchema, req)

    click.echo("Meta:")
    ctx.dump_proto_message(response.meta)

    click.echo("Annotations:")
    ctx.dump_proto_message(response.annotations)

    with output_path.open("wb") as f:
        schema: google.protobuf.message.Message = response.schema
        f.write(schema.SerializeToString())
    click.echo(f"Schema dumped to : {str(output_path)}")


@schema_registry_subcommand.command(name="put", help="Put schema")
@click.argument(
    "schema_path",
    metavar="<file path>",
    type=click.Path(exists=True, dir_okay=False, path_type=pathlib.Path)
)
@click.option(
    "--annotations",
    "-a",
    "annotations",
    metavar="<json>",
    type=click.STRING,
)
def put_schema(schema_path: pathlib.Path, annotations: typing.Optional[str]):

    fds = descriptor_pb2.FileDescriptorSet()
    fds.ParseFromString(schema_path.read_bytes())

    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)

    annotations_obj = {} if annotations is None else json.loads(annotations)
    response: schema_service.CreateSchemaResponse = ctx.register_schema(fds, annotations_obj)
    if ctx.output == consts.Output.TABLE:
        click.echo(f"Schema created: {response.hash}")
        click.echo("Meta:")
        ctx.dump_proto_message(response.meta)

        click.echo("Annotations:")
        ctx.dump_proto_message(response.annotations)
    else:
        ctx.dump_proto_message(response)
