# coding=utf-8
import click
import inject

from tasklet.experimental.cli import cmd_label
from tasklet.experimental.cli import consts
from tasklet.experimental.cli import context
from tasklet.experimental.cli import interfaces
from tasklet.experimental.cli import tasklet_descriptor
import tasklet.api.v2.tasklet_service_pb2 as tasklet_service


@click.group(name="tasklet", cls=context.GroupWithContextOptions, help="List, create or get tasklets")
@context.option_cluster
@context.option_output
def tasklet_subcommand():
    pass


@tasklet_subcommand.command(name="list", help="Show tasklets in namespace")
@context.tasklet_descriptor_argument(argument_type=context.TaskletArgumentType.NAMESPACE)
def list_tasklets(namespace: str):
    list_tasklets_impl(namespace=namespace)


def list_tasklets_impl(namespace: str):
    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)
    # TODO: add paging
    response: tasklet_service.ListTaskletsResponse = ctx.execute_request(
        ctx.driver.get_tasklet_client().ListTasklets, tasklet_service.ListTaskletsRequest(namespace=namespace)
    )
    ctx.dump_proto_message(response)


@tasklet_subcommand.command(name="get", help="Get tasklet description")
@context.tasklet_descriptor_argument(argument_type=context.TaskletArgumentType.TASKLET)
def get_tasklet(namespace: str, tasklet: str):
    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)
    response: tasklet_service.GetTaskletResponse = ctx.execute_request(
        ctx.driver.get_tasklet_client().GetTasklet,
        tasklet_service.GetTaskletRequest(namespace=namespace, tasklet=tasklet),
    )
    ctx.dump_proto_message(response)
    if ctx.output == consts.Output.TABLE:
        cmd_label.list_labels_impl(namespace=namespace, tasklet=tasklet)


@tasklet_subcommand.command(name="create", help="Register new tasklet (t.yaml required)")
@context.tasklet_descriptor_argument
def create_tasklet():
    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)
    # noinspection PyTypeChecker
    config: tasklet_descriptor.TaskletDescriptor = inject.instance(interfaces.ITaskletDescriptor)

    response: tasklet_service.CreateTaskletResponse = ctx.execute_request(
        ctx.driver.get_tasklet_client().CreateTasklet,
        config.make_create_tasklet_message(),
    )
    ctx.dump_proto_message(response)


@tasklet_subcommand.command(name="update", help="Update existing tasklet (t.yaml required)")
@context.tasklet_descriptor_argument
def update_tasklet():
    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)
    # noinspection PyTypeChecker
    config: tasklet_descriptor.TaskletDescriptor = inject.instance(interfaces.ITaskletDescriptor)

    client = ctx.driver.get_tasklet_client()
    tasklet_response: tasklet_service.GetTaskletResponse = ctx.execute_request(
        client.GetTasklet,
        tasklet_service.GetTaskletRequest(
            tasklet=config.state.meta.name,
            namespace=config.state.meta.namespace,
        )
    )
    # FIXME: show diff and ask yes/no
    response: tasklet_service.UpdateTaskletResponse = ctx.execute_request(
        client.UpdateTasklet,
        config.make_update_tasklet_message(tasklet_response.tasklet.spec.revision),
    )
    ctx.dump_proto_message(response)
