import logging
import os
import sys
import time
from typing import BinaryIO

import click
import inject

from library.python import svn_version

from tasklet.experimental.cli import cmd_build
from tasklet.experimental.cli import cmd_execution
from tasklet.experimental.cli import cmd_label
from tasklet.experimental.cli import cmd_namespace
from tasklet.experimental.cli import cmd_sr
from tasklet.experimental.cli import cmd_tasklet
from tasklet.experimental.cli import consts
from tasklet.experimental.cli import context
from tasklet.experimental.cli import interfaces
from tasklet.experimental.cli import sandbox
import tasklet.api.v2.data_model_pb2 as data_model
import tasklet.api.v2.tasklet_service_pb2 as tasklet_service

FORMAT = "%(process)s %(asctime)s %(levelname)s [%(name)s] %(message)s"


@click.group(context_settings={"help_option_names": ["-h", "--help"]})
@click.version_option(version=svn_version.svn_revision())
def cli():
    """
    CLI tool for tasklet operations

    See more information in docs: https://docs.yandex-team.ru/tasklets/
    """
    pass


@cli.command("run", help="Run tasklet execution with given payload")
@context.option_cluster
@context.option_output
@context.tasklet_descriptor_argument(argument_type=context.TaskletArgumentType.LABEL)
@click.argument("input_data", metavar="<path_to_payload_file>", type=click.File("rb"))
@click.option(
    "-i",
    "--input-format",
    type=click.Choice((
        consts.ProtoFormat.PROTO_JSON,
        consts.ProtoFormat.PROTO_BINARY,
    )),
    default=consts.ProtoFormat.PROTO_JSON,
    help="Input file format",
)
@click.option(
    "--account",
    metavar="<account-ID>",
    type=click.STRING,
    help="Account to use in runtime service: Sandbox group or YT account",
)
@click.option(
    "--alximik",
    is_flag=True,
    hidden=True,
    help="Do not use",
)
def run_tasklet(
    namespace: str,
    tasklet: str,
    label: str,  # FIXME: label is optional
    input_data: BinaryIO,
    input_format: consts.ProtoFormat,
    account: str,
    alximik: bool
):
    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)

    client = ctx.driver.get_tasklet_client()
    serialized_input = input_data.read()
    if input_format != consts.ProtoFormat.PROTO_BINARY:
        get_label_response: tasklet_service.GetLabelResponse = ctx.execute_request(
            client.GetLabel,
            tasklet_service.GetLabelRequest(namespace=namespace, tasklet=tasklet, label=label),
        )
        schema, fds = cmd_execution.get_build_schema(get_label_response.label.spec.build_id, ctx, client)
        input_message = schema.input_message

        from tasklet.experimental.cli import proto_support

        formatter = proto_support.ProtoSupport.from_fds(fds)
        if input_format == consts.ProtoFormat.PROTO_JSON:
            serialized_input = formatter.convert_json_to_protobuf_bytes(input_message, serialized_input)
        else:
            raise RuntimeError(f"Unexpected format {input_format}")

    execute_response: tasklet_service.ExecuteResponse = ctx.execute_request(
        client.Execute,
        tasklet_service.ExecuteRequest(
            namespace=namespace,
            tasklet=tasklet,
            label=label,
            input=data_model.ExecutionInput(
                serialized_data=serialized_input,
            ),
            requirements=data_model.ExecutionRequirements(
                account_id=account,
            ),
        )
    )
    ctx.dump_proto_message(execute_response)

    if not alximik:
        return

    execution_id = execute_response.execution.meta.id
    while True:
        get_response: tasklet_service.GetExecutionResponse = ctx.execute_request(
            client.GetExecution,
            tasklet_service.GetExecutionRequest(id=execution_id)
        )
        ctx.dump_proto_message(get_response)
        click.echo("=======")
        time.sleep(1)
        if get_response.execution.status.status == data_model.E_EXECUTION_STATUS_FINISHED:
            ts = get_response.execution.meta.created_at.ToDatetime()
            te = get_response.execution.status.stats.finished_at.ToDatetime()
            duration = te - ts
            click.echo(f"DURATION: {str(duration)}")
            break


@cli.command("upload", help="Upload tasklet build to Sandbox")
@sandbox.upload_arguments
def upload(
    description: str,
    attributes: [str],
    arch: str,
    resource_type: str,
    owner: str,
    file_meta: "common_upload.MDSHandle.FileMeta"
) -> int:
    # noinspection PyTypeChecker
    ctx: context.TaskletContext = inject.instance(interfaces.ITaskletContext)

    return sandbox.upload_resource(ctx.token, description, attributes, arch, resource_type, owner, file_meta)


def main():
    if os.environ.get("TASKLET_DEBUG", "") != "":
        level = logging.DEBUG
    else:
        level = logging.CRITICAL

    logger = logging.getLogger()
    logger.setLevel(level)
    stdout_handler = logging.StreamHandler(sys.stderr)
    stdout_handler.setFormatter(logging.Formatter(FORMAT))
    stdout_handler.setLevel(level)
    logger.addHandler(stdout_handler)

    def setup_injector(binder: inject.Binder):
        def descriptor_constructor() -> "tasklet.experimental.cli.tasklet_descriptor.TaskletDescriptor":
            from tasklet.experimental.cli import tasklet_descriptor

            return tasklet_descriptor.TaskletDescriptor()

        binder \
            .bind(interfaces.ITaskletContext, context.TaskletContext()) \
            .bind_to_constructor(interfaces.ITaskletDescriptor, descriptor_constructor)

    inject.clear_and_configure(setup_injector)

    cli.add_command(cmd_namespace.namespace_subcommand, name="namespace")
    cli.add_command(cmd_tasklet.tasklet_subcommand, name="tasklet")
    cli.add_command(cmd_build.build_subcommand, name="build")
    cli.add_command(cmd_label.label_subcommand, name="label")
    cli.add_command(cmd_execution.execution_subcommand, name="execution")
    cli.add_command(cmd_sr.schema_registry_subcommand, name="sr")
    cli()


if __name__ == '__main__':
    main()
